<?php

namespace ASENHA\Classes;

/**
 * Class for Enhance List Tables module
 *
 * @since 6.9.5
 */
class Enhance_List_Tables {
    
    /**
     * Current post type. For Content Admin >> Show Custom Taxonomy Filters functionality.
     */
    public $post_type;

    /**
     * Show featured images column in list tables for pages and post types that support featured image
     *
     * @since 1.0.0
     */
    public function show_featured_image_column() {

        $post_types = get_post_types( array( 'public' => true ), 'names' );

        foreach ( $post_types as $post_type_key => $post_type_name ) {

            if ( post_type_supports( $post_type_key, 'thumbnail' ) ) {

                add_filter( "manage_{$post_type_name}_posts_columns",[ $this, 'add_featured_image_column' ], 999 );
                add_action( "manage_{$post_type_name}_posts_custom_column", [ $this, 'add_featured_image' ], 10, 2 );

            }

        }

    }

    /**
     * Add a column called Featured Image as the first column
     *
     * @param mixed $columns
     * @return void
     * @since 1.0.0
     */
    public function add_featured_image_column( $columns ) {
        
        $new_columns = array();

        foreach ( $columns as $key => $value ) {

            if ( 'title' == $key) {
                
                // We add featured image column before the 'title' column
                $new_columns['asenha-featured-image'] = 'Featured Image';   

            }
            
            if ( 'thumb' == $key )  {

                // For WooCommerce products, we add featured image column before it's native thumbnail column
                $new_columns['asenha-featured-image'] = 'Product Image';    

            }

            $new_columns[$key] = $value;

        }
        
        // Replace WooCommerce thumbnail column with ASE featured image column
        if ( array_key_exists( 'thumb', $new_columns ) ) {
            unset( $new_columns['thumb'] );         
        }

        return $new_columns;

    }

    /**
     * Echo featured image's in thumbnail size to a column
     *
     * @param mixed $column_name
     * @param mixed $id
     * @since 1.0.0
     */
    public function add_featured_image( $column_name, $id ) {

        if ( 'asenha-featured-image' === $column_name ) {

            if ( has_post_thumbnail( $id ) ) {

                $size = 'thumbnail';

                echo get_the_post_thumbnail( $id, $size, '' );

            } else {

                echo '<img src="' . esc_url( plugins_url( 'assets/img/default_featured_image.jpg', __DIR__ ) ) . '" />';

            }
        }

    }

    /**
     * Show excerpt column in list tables for pages and post types that support excerpt.
     *
     * @since 1.0.0
     */
    public function show_excerpt_column() {

        $post_types = get_post_types( array( 'public' => true ), 'names' );

        foreach ( $post_types as $post_type_key => $post_type_name ) {

            if ( post_type_supports( $post_type_key, 'excerpt' ) ) {

                add_filter( "manage_{$post_type_name}_posts_columns",[ $this, 'add_excerpt_column' ] );
                add_action( "manage_{$post_type_name}_posts_custom_column", [ $this, 'add_excerpt' ], 10, 2 );

            }

        }

    }

    /**
     * Add a column called Featured Image as the first column
     *
     * @param mixed $columns
     * @return void
     * @since 1.0.0
     */
    public function add_excerpt_column( $columns ) {

        $new_columns = array();

        foreach ( $columns as $key => $value ) {

            $new_columns[$key] = $value;

            if ( $key == 'title' ) {

                $new_columns['asenha-excerpt'] = 'Excerpt'; 

            }

        }

        return $new_columns;

    }

    /**
     * Echo featured image's in thumbnail size to a column
     *
     * @param mixed $column_name
     * @param mixed $id
     * @since 1.0.0
     */
    public function add_excerpt( $column_name, $id ) {

        if ( 'asenha-excerpt' === $column_name ) {

            $excerpt = get_the_excerpt( $id ); // about 310 characters
            $excerpt = substr( $excerpt, 0, 160 ); // truncate to 160 characters
            $short_excerpt = substr( $excerpt, 0, strrpos( $excerpt, ' ' ) );

            echo wp_kses_post( $short_excerpt );

        }

    }

    /**
     * Add ID column list table of pages, posts, custom post types, media, taxonomies, custom taxonomies, users amd comments
     *
     * @since 1.0.0
     */
    public function show_id_column() {

        // For pages and hierarchical post types list table

        add_filter( 'manage_pages_columns', [ $this, 'add_id_column' ] );
        add_action( 'manage_pages_custom_column', [ $this, 'add_id_echo_value' ], 10, 2 );              

        // For posts and non-hierarchical custom posts list table

        add_filter( 'manage_posts_columns', [ $this, 'add_id_column' ] );
        add_action( 'manage_posts_custom_column', [ $this, 'add_id_echo_value' ], 10, 2 );

        // For media list table

        add_filter( 'manage_media_columns', [ $this, 'add_id_column' ] );
        add_action( 'manage_media_custom_column', [ $this, 'add_id_echo_value' ], 10, 2 );

        // For list table of all taxonomies

        $taxonomies = get_taxonomies( [ 'public' => true ], 'names' );

        foreach ( $taxonomies as $taxonomy ) {
            
            add_filter( 'manage_edit-' . $taxonomy . '_columns', [ $this, 'add_id_column' ] );
            add_action( 'manage_' . $taxonomy . '_custom_column', [ $this, 'add_id_return_value' ], 10, 3 );

        }

        // For users list table

        add_filter( 'manage_users_columns', [ $this, 'add_id_column' ]);    
        add_action( 'manage_users_custom_column', [ $this, 'add_id_return_value' ], 10, 3 );

        // For comments list table

        add_filter( 'manage_edit-comments_columns', [ $this, 'add_id_column' ]);
        add_action( 'manage_comments_custom_column', [ $this, 'add_id_echo_value' ], 10, 3 );

    }

    /**
     * Add a column called ID
     *
     * @param mixed $columns
     * @return void
     * @since 1.0.0
     */
    public function add_id_column( $columns ) {

        $columns['asenha-id'] = 'ID';

        return $columns;

    }

    /**
     * Echo post ID value to a column
     *
     * @param mixed $column_name
     * @param mixed $id
     * @since 1.0.0
     */
    public function add_id_echo_value( $column_name, $id ) {

        if ( 'asenha-id' === $column_name ) {
            echo esc_html( $id );
        }

    }

    /**
     * Return post ID value to a column
     *
     * @param mixed $value
     * @param mixed $column_name
     * @param mixed $id
     * @since 1.0.0
     */
    public function add_id_return_value( $value, $column_name, $id ) {

        if ( 'asenha-id' === $column_name ) {
            $value = $id;
        }

        return $value;

    }

    /**
     * Add file size column to media library
     *
     * @since 6.9.5
     */
    public function add_column_file_size( $columns ) {
        $columns['asenha-file-size'] = 'File Size';
        return $columns;        
    }
    
    /**
     * Display the file size value
     *
     * @since 6.9.5
     */
    public function display_file_size( $column_name, $attachment_id ) {
        if ( 'asenha-file-size' != $column_name ) {
            return;
        }

        $file_size = filesize( get_attached_file( $attachment_id ) );
        $file_size = size_format( $file_size, 1 ); // Show one decimal point
        echo esc_html( $file_size );
    }

    /**
     * Add file size column to media library
     *
     * @since 6.9.5
     */
    public function add_media_styles() {
        echo '<style>.column-asenha-file-siz {width: 60px;}</style>';
    }

    /**
     * Add ID in the action row of list tables for pages, posts, custom post types, media, taxonomies, custom taxonomies, users amd comments
     *
     * @since 4.7.4
     */
    public function show_id_in_action_row() {

        add_filter( 'page_row_actions', array( $this, 'add_id_in_action_row' ), 10, 2 );
        add_filter( 'post_row_actions', array( $this, 'add_id_in_action_row' ), 10, 2 );
        add_filter( 'cat_row_actions', array( $this, 'add_id_in_action_row' ), 10, 2 );
        add_filter( 'tag_row_actions', array( $this, 'add_id_in_action_row' ), 10, 2 );
        add_filter( 'media_row_actions', array( $this, 'add_id_in_action_row' ), 10, 2 );
        add_filter( 'comment_row_actions', array( $this, 'add_id_in_action_row' ), 10, 2 );
        add_filter( 'user_row_actions', array( $this, 'add_id_in_action_row' ), 10, 2 );

    }

    /**
     * Output the ID in the action row
     *
     * @since 4.7.4
     */
    public function add_id_in_action_row( $actions, $object ) {

        if ( current_user_can( 'edit_posts' ) ) {

            // For pages, posts, custom post types, media/attachments, users
            if ( property_exists( $object, 'ID' ) ) {
                $id = $object->ID;
            }

            // For taxonomies
            if ( property_exists( $object, 'term_id' ) ) {
                $id = $object->term_id;
            }

            // For comments
            if ( property_exists( $object, 'comment_ID' ) ) {
                $id = $object->comment_ID;
            }

            $actions['asenha-list-table-item-id'] = '<span class="asenha-list-table-item-id">ID: ' . $id . '</span>';

        }

        return $actions;

    }

    /**
     * Hide comments column in list tables for pages, post types that support comments, and alse media/attachments.
     *
     * @since 1.0.0
     */
    public function hide_comments_column() {

        $post_types = get_post_types( array( 'public' => true ), 'names' );

        foreach ( $post_types as $post_type_key => $post_type_name ) {

            if ( post_type_supports( $post_type_key, 'comments' ) ) {

                if ( 'attachment' != $post_type_name ) {

                    // For list tables of pages, posts and other post types
                    add_filter( "manage_{$post_type_name}_posts_columns", [ $this, 'remove_comment_column' ] );

                } else {

                    // For list table of media/attachment
                    add_filter( 'manage_media_columns', [ $this, 'remove_comment_column' ] );

                }

            }

        }

    }

    /**
     * Add a column called ID
     *
     * @param mixed $columns
     * @return void
     * @since 1.0.0
     */
    public function remove_comment_column( $columns ) {

        unset( $columns['comments'] );

        return $columns;

    }

    /**
     * Hide tags column in list tables for posts.
     *
     * @since 1.0.0
     */
    public function hide_post_tags_column() {

        $post_types = get_post_types( array( 'public' => true ), 'names' );

        foreach ( $post_types as $post_type_key => $post_type_name ) {

            if ( $post_type_name == 'post' ) {

                add_filter( "manage_posts_columns", [ $this, 'remove_post_tags_column' ] );

            }

        }

    }

    /**
     * Add a column called ID
     *
     * @param mixed $columns
     * @return void
     * @since 1.0.0
     */
    public function remove_post_tags_column( $columns ) {

        unset( $columns['tags'] );

        return $columns;

    }
    
    /**
     * Show custom (hierarchical) taxonomy filter(s) for all post types.
     *
     * @since 1.0.0
     */
    public function show_custom_taxonomy_filters( $post_type ) {

        $post_taxonomies = get_object_taxonomies( $post_type, 'objects' );

        // Only show custom taxonomy filters for post types other than 'post'

        if ( 'post' != $post_type && 'attachment' != $post_type && 'asenha_code_snippet' != $post_type ) {

            array_walk( $post_taxonomies, [ $this, 'output_taxonomy_filter' ] );

        }

    }

    /**
     * Output filter on the post type's list table for a taxonomy
     *
     * @since 1.0.0
     */
    public function output_taxonomy_filter( $post_taxonomy ) {

        // Only show taxonomy filter when the taxonomy is hierarchical

        if ( true === $post_taxonomy->hierarchical ) {

            wp_dropdown_categories( array(
                'show_option_all'   => sprintf( 'All %s', $post_taxonomy->label ),
                'orderby'           => 'name',
                'order'             => 'ASC',
                'hide_empty'        => false,
                'hide_if_empty'     => true,
                'selected'          => sanitize_text_field( ( isset( $_GET[$post_taxonomy->query_var] ) && ! empty( $_GET[$post_taxonomy->query_var] ) ) ? $_GET[$post_taxonomy->query_var] : '' ), 
                'hierarchical'      => true,
                'name'              => $post_taxonomy->query_var,
                'taxonomy'          => $post_taxonomy->name,
                'value_field'       => 'slug',
            ) );

        }

    }
            
}