<?php
/**
 * Plugin Name: AudioIgniter Pro
 * Plugin URI: http://www.cssigniter.com/plugins/audioigniter-pro/
 * Description: Pro add-ons for AudioIgniter.
 * Author: The CSSIgniter Team
 * Author URI: http://www.cssigniter.com
 * Version: 1.7.0
 * Text Domain: audioigniter-pro
 * Domain Path: languages
 *
 * AudioIgniter Pro is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * AudioIgniter Pro is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with AudioIgniter Pro. If not, see <http://www.gnu.org/licenses/>.
 *
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * AudioIgniter Pro plugin
 */
class AudioIgniter_Pro {

	/**
	 * AudioIgniter Pro version.
	 *
	 * @var string
	 * @since 1.0.0
	 * @since 1.6.0 Changed from static to non-static.
	 */
	public $version = null;

	/**
	 * @var AudioIgniter_Icons
	 * @since 1.7.0
	 */
	private $icons = null;

	/**
	 * Instance of this class.
	 *
	 * @var AudioIgniter_Pro
	 * @since 1.0.1
	 */
	protected static $instance = null;

	/**
	 * AudioIgniter Pro Instance.
	 *
	 * Instantiates or reuses an instance of AudioIgniter Pro.
	 *
	 * @since 1.0.0
	 * @static
	 * @see AudioIgniter_Pro()
	 * @return AudioIgniter_Pro - Single instance.
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * The URL directory path (with trailing slash) of the main plugin file.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	protected static $plugin_url = '';

	/**
	 * The filesystem directory path (with trailing slash) of the main plugin file.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	protected static $plugin_path = '';


	/**
	 * AudioIgniter constructor. Intentionally left empty so that instances can be created without
	 * re-loading of resources (e.g. scripts/styles), or re-registering hooks.
	 * http://wordpress.stackexchange.com/questions/70055/best-way-to-initiate-a-class-in-a-wp-plugin
	 * https://gist.github.com/toscho/3804204
	 *
	 * @since 1.0.0
	 */
	public function __construct() {}


	/**
	 * Kickstarts plugin loading.
	 *
	 * @since 1.0.0
	 */
	public function plugin_setup() {
		if ( is_null( $this->version ) ) {
			if ( ! function_exists( 'get_plugin_data' ) ) {
				include_once ABSPATH . 'wp-admin/includes/plugin.php';
			}
			$plugin_data = get_plugin_data( __FILE__ );

			$this->version = $plugin_data['Version'];
		}

		self::$plugin_url  = plugin_dir_url( __FILE__ );
		self::$plugin_path = plugin_dir_path( __FILE__ );

		load_plugin_textdomain( 'audioigniter-pro', false, dirname( self::plugin_basename() ) . '/languages' );

		add_action( 'audioigniter_init', array( $this, 'init' ) );
		add_action( 'audioigniter_admin_init', array( $this, 'admin_init' ) );

		if ( ! class_exists( 'AudioIgniter' ) ) {
			add_action( 'admin_notices', 'AudioIgniter_Pro::notice_plugin_required' );
		}

		if ( defined( 'ELEMENTOR_VERSION' ) ) {
			add_action( 'elementor/init', array( $this, 'add_element' ) );
			add_action( 'elementor/init', array( $this, 'add_element_category' ) );
			add_action( 'elementor/preview/enqueue_scripts', array( $this, 'element_scripts' ) );
		}
	}

	/**
	 * Registers actions that need to be run on both admin and frontend
	 *
	 * @since 1.0.0
	 */
	public function init() {
		add_action( 'init', array( $this, 'register_taxonomies' ) );
		add_action( 'init', array( $this, 'register_scripts' ) );
		add_action( 'init', array( $this, 'register_shortcodes' ) );
		add_action( 'vc_before_init', array( $this, 'register_vc_shortcodes' ) );
		add_filter( 'audioigniter_register_widgets', array( $this, 'register_widgets' ) );

		require_once 'class-audioigniter-customizer.php';
		new AudioIgniter_Customizer();

		require_once 'class-audioigniter-icons.php';
		$this->icons = new AudioIgniter_Icons();

		add_action( 'customize_preview_init', array( $this, 'customize_preview_js' ) );

		require_once 'block/src/block.php';

		add_filter( 'audioigniter_shortcode_data_attributes_array', array( $this, 'filter_shortcode_data_attributes_array' ), 10, 3 );
		add_filter( 'audioigniter_shortcode_shortcircuit', array( $this, 'filter_shortcode_shortcircuit' ), 10, 4 );
		add_filter( 'audioigniter_player_types', array( $this, 'filter_player_types' ) );
		add_filter( 'audioigniter_metabox_tracks_show_upgrade_button', '__return_false' );

		add_filter( 'audioigniter_default_track_values', array( $this, 'filter_default_track_values' ) );
		add_filter( 'audioigniter_playlist_endpoint_track', array( $this, 'filter_playlist_endpoint_track' ), 10, 4 );
		add_filter( 'audioigniter_get_playlist_data_attributes_array', array( $this, 'filter_get_playlist_data_attributes_array' ), 10, 2 );

		add_filter( 'register_post_type_args', array( $this, 'fiter_register_post_type_args' ), 10, 2 );

		add_action( 'wp_footer', array( $this, 'global_footer_player' ) );
	}

	/**
	 * Registers actions that need to be run on admin only.
	 *
	 * @since 1.0.0
	 */
	public function admin_init() {
		if ( ! is_admin() ) {
			return;
		}

		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );

		add_filter( 'audioigniter_metabox_tracks_container_classes', array( $this, 'filter_metabox_tracks_container_classes' ) );
		add_filter( 'audioigniter_metabox_track_classes', array( $this, 'filter_metabox_track_classes' ), 10, 2 );
		add_filter( 'audioigniter_metabox_tracks_footer_links', array( $this, 'filter_metabox_tracks_footer_links' ), 10 );
		add_action( 'audioigniter_metabox_tracks_field_controls', array( $this, 'action_metabox_tracks_field_controls' ), 10, 2 );
		add_action( 'audioigniter_metabox_tracks_repeatable_track_field_before_title', array( $this, 'action_metabox_tracks_move_handle' ) );
		add_action( 'audioigniter_metabox_tracks_repeatable_track_field_after_track_upload_button', array( $this, 'action_metabox_tracks_soundcloud_sync_button' ) );
		add_action( 'audioigniter_metabox_tracks_repeatable_track_field_after_download_url_button', array( $this, 'action_metabox_tracks_download_url_checkbox' ), 10, 2 );
		add_action( 'audioigniter_metabox_tracks_repeatable_track_fields_column_2', array( $this, 'action_metabox_tracks_repeatable_lyrics' ), 30, 2 );
		add_action( 'audioigniter_metabox_settings_group_player_track_track_listing_repeat_fields', array( $this, 'action_metabox_settings_track_allow_loop' ), 30, 2 );
		add_action( 'audioigniter_metabox_settings_group_player_track_listing_fields', array( $this, 'action_metabox_settings_skip_amount' ), 30, 2 );
		add_action( 'audioigniter_metabox_settings_group_player_track_listing_fields', array( $this, 'action_metabox_settings_shuffle' ), 30, 2 );
		add_action( 'audioigniter_metabox_settings_group_player_track_listing_fields', array( $this, 'action_metabox_settings_allow_playback_rate' ), 30, 2 );
		add_action( 'audioigniter_metabox_settings_group_player_track_listing_fields', array( $this, 'action_metabox_settings_timer_countdown_by_default' ), 30, 2 );
		add_action( 'audioigniter_metabox_settings_group_tracks_fields', array( $this, 'action_metabox_settings_stop_on_track_finish' ), 30, 2 );
		add_action( 'audioigniter_metabox_settings_group_tracks_fields', array( $this, 'action_metabox_settings_remember_last_player_position' ), 30, 2 );
		add_action( 'audioigniter_metabox_settings_group_tracks_fields', array( $this, 'action_metabox_settings_initial_track' ), 30, 2 );
		add_action( 'audioigniter_metabox_settings_group_tracks_fields', array( $this, 'action_metabox_settings_tracks_delay' ), 30, 2 );

		add_action( 'wp_ajax_ai_upload_image', array( $this, 'ajax_upload_image' ) );

		add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ) );

		add_action( 'audioigniter_save_post', array( $this, 'action_save_post' ) );
		add_filter( 'audioigniter_sanitize_playlist_track', array( $this, 'filter_sanitize_playlist_track' ), 10, 4 );

		add_action( 'save_post', array( $this, 'invalidate_transient_global_footer_player_id' ) );
		add_action( 'delete_post', array( $this, 'invalidate_transient_global_footer_player_id' ) );

		add_filter( 'block_categories_all', array( $this, 'block_categories' ), 10, 2 );

		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_editor_assets' ) );
	}

	public function register_taxonomies() {
		$labels = array(
			'name'              => esc_html_x( 'Playlist Categories', 'taxonomy general name', 'audioigniter-pro' ),
			'singular_name'     => esc_html_x( 'Playlist Category', 'taxonomy singular name', 'audioigniter-pro' ),
			'search_items'      => esc_html__( 'Search Playlist Categories', 'audioigniter-pro' ),
			'all_items'         => esc_html__( 'All Playlist Categories', 'audioigniter-pro' ),
			'parent_item'       => esc_html__( 'Parent Playlist Categories', 'audioigniter-pro' ),
			'parent_item_colon' => esc_html__( 'Parent Playlist Categories:', 'audioigniter-pro' ),
			'edit_item'         => esc_html__( 'Edit Playlist Category', 'audioigniter-pro' ),
			'update_item'       => esc_html__( 'Update Playlist Category', 'audioigniter-pro' ),
			'add_new_item'      => esc_html__( 'Add New Playlist Category', 'audioigniter-pro' ),
			'new_item_name'     => esc_html__( 'New Playlist Category Name', 'audioigniter-pro' ),
			'menu_name'         => esc_html__( 'Categories', 'audioigniter-pro' ),
			'view_item'         => esc_html__( 'View Playlist Category', 'audioigniter-pro' ),
			'popular_items'     => esc_html__( 'Popular Playlist Categories', 'audioigniter-pro' ),
		);

		register_taxonomy( 'ai_playlist_category', array( AudioIgniter()->post_type ), array(
			'labels'            => $labels,
			'hierarchical'      => true,
			'public'            => false,
			'show_ui'           => true,
			'show_admin_column' => true,
			'show_in_menu'      => true,
			'show_in_nav_menus' => false,
			'rewrite'           => array( 'slug' => _x( 'playlist-category', 'taxonomy slug', 'audioigniter-pro' ) ),
			'description'       => esc_html__( 'Used solely for better playlist organization.', 'audioigniter-pro' ),
		) );
	}

	public function register_vc_shortcodes() {
		require_once 'class-audioigniter-visual-composer-playlist.php';
		$vc_shortcode = new AudioIgniter_Visual_Composer_Playlist();
		$vc_shortcode->load();
	}

	public function register_widgets( $widgets ) {
		return array_merge( $widgets, array(
			'AudioIgniter_Playlist_Widget' => $this->plugin_path() . 'class-audioigniter-playlist-widget.php',
		) );
	}

	public function register_shortcodes() {
		add_shortcode( 'ai_track', array( $this, 'shortcode_ai_track' ) );
	}

	/**
	 * Returns the output of the [ai_track] shortcode.
	 *
	 * @since 1.7.0
	 *
	 * @param array  $atts    The shortcode attributes.
	 * @param string $content Content, when used with a shortcode closing tag.
	 * @param string $tag     The shortcode name used to reach this function.
	 *
	 * @return string
	 */
	public function shortcode_ai_track( $atts, $content, $tag ) {
		if ( empty( $atts['audio'] ) ) {
			return '';
		}

		$has_download = false === array_search( 'download', (array) $atts, true ) ? false : true;

		$atts = shortcode_atts( array(
			'id'       => '',
			'class'    => '',
			'audio'    => '',
			'title'    => '',
			'subtitle' => '',
			'buy_url'  => '',
			'download' => '',
			'cover'    => '',
			'lyrics'   => '',
			'player'   => 'simple', // Player type. 'full', 'simple'
		), $atts, $tag );

		$id         = $atts['id'];
		$class_name = $atts['class'];

		$download_url = '';
		if ( $has_download && empty( $atts['download'] ) ) {
			$download_url = $atts['audio'];
		} elseif ( ! empty( $atts['download'] ) ) {
			$download_url = $atts['download'];
		}

		$track = array(
			'title'            => $atts['title'],
			'subtitle'         => $atts['subtitle'],
			'audio'            => $atts['audio'],
			'buyUrl'           => $atts['buy_url'],
			'downloadUrl'      => $download_url,
			'downloadFilename' => AudioIgniter()->get_filename_from_url( $download_url ),
			'cover'            => $atts['cover'],
			'lyrics'           => $atts['lyrics'],
			'player'           => 'full' === $atts['player'] ? 'full' : 'simple',
		);

		if ( ! $track['downloadFilename'] ) {
			$track['downloadFilename'] = $track['downloadUrl'];
		}

		$params = array(
			'data-track'                  => wp_json_encode( $track ),
			'data-display-active-cover'   => $atts['cover'] ? 'true' : 'false',
			'data-display-credits'        => 'false',
			'data-allow-playback-rate'    => 'false',
			'data-display-buy-buttons'    => 'true',
			'data-volume'                 => 100,
			'data-skip-amount'            => 0,
			'data-stop-on-finish'         => 'true',
			'data-timer-countdown'        => 'false',
			'data-allow-tracklist-loop'   => 'true',
			'data-max-width'              => '',
			'data-remember-last'          => 'false',
			'data-display-artist-names'   => 'true',
			'data-player-type'            => $track['player'],
			'data-tracks-url'             => '',
			'data-display-tracklist'      => 'false',
			'data-allow-tracklist-toggle' => 'true',
			'data-allow-track-loop'       => 'false',
			'data-display-track-no'       => 'false',
			'data-buy-buttons-target'     => 'true',
			'data-cycle-tracks'           => 'false',
			'data-limit-tracklist-height' => 'false',
			'data-tracklist-height'       => '',
			'data-reverse-track-order'    => 'false',
			'data-initial-track'          => 1,
			'data-tracks-delay'           => 0,
			'data-shuffle'                => 'false',
			'data-shuffle-default'        => 'false',
		);

		$data_attr = '';
		foreach ( $params as $attribute => $value ) {
			$data_attr .= sprintf( '%s="%s" ', sanitize_key( $attribute ), esc_attr( $value ) );
		}

		$player_classes = array_merge( array(
			'audioigniter-root',
		), explode( ' ', $class_name ) );

		return sprintf( '<div id="audioigniter-%s" class="%s" %s></div>',
			esc_attr( $id ),
			esc_attr( implode( ' ', $player_classes ) ),
			$data_attr
		);
	}

	/**
	 * Register (but not enqueue) all scripts and styles to be used throughout the plugin.
	 *
	 * @since 1.0.0
	 */
	public function register_scripts() {
		wp_register_script( 'audioigniter-pro-admin', $this->plugin_url() . 'assets/js/audioigniter-pro.js', array(
			'jquery-ui-core',
			'jquery-ui-sortable',
		), $this->version, true );

		wp_localize_script( 'audioigniter-pro-admin', 'ai_pro_scripts', array(
			'ajax_url'         => admin_url( 'admin-ajax.php' ),
			'_ai_upload_nonce' => wp_create_nonce( 'audioigniter-image-upload' ),
			'sc_client_id'     => get_option( 'audioigniter_sc_client_id' ),
			'messages'         => array(
				'media_title_batch'                 => esc_html__( 'Select or upload audio media', 'audioigniter-pro' ),
				'error_intro'                       => esc_html__( 'There was an error while fetching data from SoundCloud.', 'audioigniter-pro' ),
				'error_missing_client_id'           => esc_html__( 'Make sure your SoundCloud client ID is set in the customizer.', 'audioigniter-pro' ),
				'error_disabled_embed'              => esc_html__( 'Most probably the track owner has disabled external embedding.', 'audioigniter-pro' ),
				'error_track_support'               => esc_html__( 'Currently only single tracks are supported by the AudioIgniter player.', 'audioigniter-pro' ),
				// There are multiple occurrences of this string. Make sure to change it throughout.
				'download_audio_url_link'           => esc_html__( 'Sync download URL with audio', 'audioigniter-pro' ),
				// There are multiple occurrences of this string. Make sure to change it throughout.
				'download_audio_url_unlink'         => esc_html__( 'Unsync download URL from audio', 'audioigniter-pro' ),
				'download_audio_url_link_confirm'   => esc_html__( 'Are you sure you want to link download and audio URLs on all tracks?', 'audioigniter-pro' ),
				'download_audio_url_unlink_confirm' => esc_html__( 'Are you sure you want to remove the link between the download and audio URLs on all tracks?', 'audioigniter-pro' ),
			),
		) );

		wp_localize_script( 'audioigniter', 'ai_pro_front_scripts', array(
			'multi_sound_disabled' => true,
			'typography_disabled'  => get_theme_mod( 'audioigniter_disable_typography', '' ),
		) );

		wp_register_script( 'audioigniter-pro-block-editor', untrailingslashit( $this->plugin_url() ) . '/block/build/block.js', array(
			'wp-components',
			'wp-blocks',
			'wp-element',
			'wp-block-editor',
			'wp-data',
			'wp-date',
			'wp-i18n',
			'wp-compose',
			'wp-keycodes',
			'wp-html-entities',
			'wp-server-side-render',
			'audioigniter',
		), $this->version, true );

		wp_register_style( 'audioigniter-pro-block-editor', $this->plugin_url() . 'block/build/block.css', array(
			'wp-edit-blocks',
			'audioigniter',
		), $this->version );

		wp_localize_script( 'audioigniter-pro-block-editor', 'aiColors', array(
			'disableTypography'           => get_theme_mod( 'audioigniter_disable_typography', '' ),
			'backgroundColor'             => get_theme_mod( 'audioigniter_bg_color' ),
			'bg_image'                    => get_theme_mod( 'audioigniter_bg_image' ),
			'bg_image_repeat'             => get_theme_mod( 'audioigniter_bg_image_repeat' ),
			'bg_image_position'           => get_theme_mod( 'audioigniter_bg_image_position' ),
			'bg_image_cover'              => get_theme_mod( 'audioigniter_bg_image_cover' ),
			'textColor'                   => get_theme_mod( 'audioigniter_text_color' ),
			'accentColor'                 => get_theme_mod( 'audioigniter_accent_color' ),
			'textOnAccentColor'           => get_theme_mod( 'audioigniter_text_on_accent_color' ),
			'controlColor'                => get_theme_mod( 'audioigniter_control_color' ),
			'playerTextColor'             => get_theme_mod( 'audioigniter_player_text_color' ),
			'playerButtonBackgroundColor' => get_theme_mod( 'audioigniter_player_button_background_color' ),
			'playerButtonTextColor'       => get_theme_mod( 'audioigniter_player_button_text_color' ),
			'playerButtonActiveColor'     => get_theme_mod( 'audioigniter_player_button_active_background_color' ),
			'playerButtonActiveTextColor' => get_theme_mod( 'audioigniter_player_button_active_text_color' ),
			'trackBarColor'               => get_theme_mod( 'audioigniter_track_bar_color' ),
			'progressBarColor'            => get_theme_mod( 'audioigniter_progress_bar_color' ),
			'trackBackgroundColor'        => get_theme_mod( 'audioigniter_track_background_color' ),
			'trackTextColor'              => get_theme_mod( 'audioigniter_track_text_color' ),
			'activeTrackBackgroundColor'  => get_theme_mod( 'audioigniter_track_active_background_color' ),
			'trackActiveTextColor'        => get_theme_mod( 'audioigniter_track_active_text_color' ),
			'trackButtonBackgroundColor'  => get_theme_mod( 'audioigniter_track_button_background_color' ),
			'trackButtonTextColor'        => get_theme_mod( 'audioigniter_track_button_text_color' ),
			'lyricsModalBackgroundColor'  => get_theme_mod( 'audioigniter_lyrics_modal_background_color' ),
			'lyricsModalTextColor'        => get_theme_mod( 'audioigniter_lyrics_modal_text_color' ),
		) );
	}

	/**
	 * Registers metaboxes for the ai_playlist post type.
	 *
	 * @since 1.7.0
	 */
	public function add_meta_boxes() {
		add_meta_box( 'ai-pro-meta-box-player-buttons', esc_html__( 'Player Buttons', 'audioigniter-pro' ), array( $this, 'metabox_player_buttons' ), AudioIgniter()->post_type, 'normal', 'default' );
	}

	/**
	 * Enqueues admin scripts and styles.
	 *
	 * @since 1.0.0
	 */
	public function enqueue_admin_scripts( $hook ) {
		$screen = get_current_screen();

		if ( 'post' === $screen->base && AudioIgniter()->post_type === $screen->post_type ) {
			wp_enqueue_script( 'audioigniter-pro-admin' );
		}
	}

	/**
	 * Binds JS handlers to make Customizer preview reload changes asynchronously.
	 *
	 * @since 1.0.0
	 */
	public function customize_preview_js() {
		wp_enqueue_script( 'audioigniter-customizer-preview', untrailingslashit( $this->plugin_url() ) . '/assets/js/admin/customizer-preview.js', array(
			'jquery',
		), $this->version, true );
	}

	/**
	 * Enqueues editor scripts and styles.
	 *
	 * @since 1.7.0
	 */
	public function enqueue_editor_assets( $hook ) {
		wp_enqueue_script( 'audioigniter-pro-block-editor' );
		wp_enqueue_style( 'audioigniter-pro-block-editor' );
	}

	/**
	 * Register AudioIgniter's block category
	 *
	 * @since 1.7.0
	 */
	public function block_categories( $categories, $post ) {
		return array_merge( $categories, array(
			array(
				'slug'  => 'audioigniter',
				'title' => __( 'AudioIgniter', 'audioigniter-pro' ),
			),
		) );
	}

	/**
	 * Registers AudioIgniter's element
	 *
	 * @since 1.5.0
	 */
	public function add_element() {
		require_once $this->plugin_path() . 'element/ai-element.php';
	}

	/**
	 * Registers AudioIgniter's element category
	 *
	 * @since 1.5.0
	 */
	public function add_element_category() {
		\Elementor\Plugin::instance()->elements_manager->add_category(
			'audioigniter-pro',
			array(
				'title' => __( 'AudioIgniter Pro', 'audioigniter-pro' ),
				'icon'  => 'font',
			)
		);
	}


	/**
	 * Enqueues AudioIgniter element's script
	 *
	 * @since 1.5.0
	 */
	public function element_scripts() {
		wp_enqueue_script( 'ai-element', $this->plugin_url() . 'assets/js/audioigniter-element.js', array( 'jquery' ), $this->version, true );
	}

	public function ajax_upload_image() {
		if ( empty( $_POST ) || ! isset( $_POST['url'] ) ) { // Input var okay.
			wp_send_json_error( __( 'Image URL is required.', 'audioigniter-pro' ) );
		}

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'audioigniter-image-upload' ) ) { // Input var okay.
			wp_send_json_error( __( 'Not authorized.', 'audioigniter-pro' ) );
		}

		global $post;
		$image_url = esc_url_raw( $_POST['url'] ); // Input var okay.
		$image_id  = media_sideload_image( $image_url, $post->ID, null, 'id' );

		if ( is_wp_error( $image_id ) ) {
			wp_send_json_error( __( 'Image could not be uploaded.', 'audioigniter-pro' ) );
		}

		$src = wp_get_attachment_thumb_url( $image_id );

		$response = array(
			'id'  => $image_id,
			'url' => $src,
		);

		wp_send_json_success( $response );
	}

	public function filter_metabox_tracks_container_classes( $classes ) {
		// Enable sortable for the repeatable tracks.
		$classes[] = 'ai-fields-sortable';

		return $classes;
	}

	public function filter_metabox_track_classes( $classes, $url ) {
		if ( strpos( $url, 'soundcloud.com' ) !== false ) {
			$classes[] = 'ai-soundcloud-track';
		}

		return $classes;
	}

	public function filter_metabox_tracks_footer_links( $links ) {
		unset( $links['rate_plugin'] );

		$new_links = array(
			'support'       => array(
				'title' => __( 'Pro Support', 'audioigniter-pro' ),
				'url'   => 'https://www.cssigniter.com/support-hub/',
			),
			'documentation' => array(
				'title' => __( 'Documentation', 'audioigniter-pro' ),
				'url'   => 'http://www.cssigniter.com/docs/audioigniter/',
			),
			'rate_plugin'   => array(
				'title' => __( 'Rate this plugin', 'audioigniter-pro' ),
				'url'   => 'https://wordpress.org/support/view/plugin-reviews/audioigniter',
			),
		);

		$links = array_merge( $links, $new_links );

		return $links;
	}

	public function action_metabox_tracks_field_controls( $location, $post_id ) {
		$tracks      = AudioIgniter()->get_post_meta( $post_id, '_audioigniter_tracks', array() );
		$all_checked = true;

		foreach ( $tracks as $track ) {
			$track = wp_parse_args( $track, AudioIgniter::get_default_track_values() );
			if ( ! $track['download_uses_track_url'] ) {
				$all_checked = false;
			}
		}

		if ( empty( $tracks ) ) {
			$all_checked = false;
		}

		?>
		<button type="button" class="button ai-add-field-batch">
			<span class="dashicons dashicons-upload"></span>
			<?php esc_html_e( 'Batch Upload', 'audioigniter-pro' ); ?>
		</button>

		<button type="button" class="button ai-use-track-url-download">
			<span class="dashicons dashicons-admin-links"></span>
			<span class="js-text">
				<?php if ( $all_checked ) {
					// There are multiple occurrences of this string. Make sure to change it throughout.
					esc_html_e( 'Unsync download URL from audio', 'audioigniter-pro' );
				} else {
					// There are multiple occurrences of this string. Make sure to change it throughout.
					esc_html_e( 'Sync download URL with audio', 'audioigniter-pro' );
				} ?>
			</span>
		</button>
		<?php
	}

	public function action_metabox_tracks_move_handle() {
		?>
		<span class="ai-field-sort-handle">
			<span class="screen-reader-text"><?php esc_html_e( 'Move track', 'audioigniter-pro' ); ?></span>
			<span class="dashicons dashicons-move"></span>
		</span>
		<?php
	}

	public function action_metabox_tracks_repeatable_lyrics( $track, $uid ) {
		?>
		<div class="ai-form-field">
			<label
				for="ai_playlist_tracks-<?php echo esc_attr( $uid ); ?>-lyrics"
				class="screen-reader-text">
				<?php esc_html_e( 'Lyrics', 'audioigniter-pro' ); ?>
			</label>
			<textarea
				id="ai_playlist_tracks-<?php echo esc_attr( $uid ); ?>-lyrics"
				class="ai-track-lyrics"
				name="ai_playlist_tracks[<?php echo esc_attr( $uid ); ?>][lyrics]"
				placeholder="<?php esc_attr_e( 'Lyrics', 'audioigniter-pro' ); ?>"
				rows="5"
			><?php echo esc_html( $track['lyrics'] ); ?></textarea>
		</div>
		<?php
	}

	public function filter_sanitize_playlist_track( $sanitized_track, $track, $post_id, $track_uid ) {
		$sanitized_track['lyrics'] = sanitize_textarea_field( $track['lyrics'] );

		return $sanitized_track;
	}

	public function action_metabox_settings_track_allow_loop( $object, $box ) {
		$track_allow_loop = AudioIgniter()->get_post_meta( $object->ID, '_audioigniter_allow_track_loop', 0 );
		?>
		<div class="ai-form-field">
			<input
				type="checkbox"
				class="ai-checkbox"
				id="_audioigniter_allow_track_loop"
				name="_audioigniter_allow_track_loop"
				value="1" <?php checked( $track_allow_loop, true ); ?>
			/>

			<label for="_audioigniter_allow_track_loop">
				<?php esc_html_e( 'Allow individual track repeat', 'audioigniter-pro' ); ?>
			</label>
		</div>
		<?php
	}

	public function action_metabox_settings_shuffle( $object, $box ) {
		$shuffle = AudioIgniter()->get_post_meta( $object->ID, '_audioigniter_shuffle', 0 );
		?>
		<div class="ai-form-field">
			<input
				type="checkbox"
				class="ai-checkbox"
				id="_audioigniter_shuffle"
				name="_audioigniter_shuffle"
				value="1" <?php checked( $shuffle, true ); ?>
			/>

			<label for="_audioigniter_shuffle">
				<?php esc_html_e( 'Enable shuffle', 'audioigniter-pro' ); ?>
			</label>
		</div>
		<?php
	}

	public function action_metabox_settings_allow_playback_rate( $object, $box ) {
		$allow_playback_rate = AudioIgniter()->get_post_meta( $object->ID, '_audioigniter_allow_playback_rate', 0 );
		?>
		<div class="ai-form-field">
			<input
				type="checkbox"
				class="ai-checkbox"
				id="_audioigniter_allow_playback_rate"
				name="_audioigniter_allow_playback_rate"
				value="1" <?php checked( $allow_playback_rate, true ); ?>
			/>

			<label for="_audioigniter_allow_playback_rate">
				<?php esc_html_e( 'Enable playback rate control', 'audioigniter-pro' ); ?>
			</label>
		</div>
		<?php
	}

	public function action_metabox_settings_timer_countdown_by_default( $object, $box ) {
		$countdown_by_default = AudioIgniter()->get_post_meta( $object->ID, '_audioigniter_countdown_by_default', 0 );
		?>
		<div class="ai-form-field">
			<input
				type="checkbox"
				class="ai-checkbox"
				id="_audioigniter_countdown_by_default"
				name="_audioigniter_countdown_by_default"
				value="1" <?php checked( $countdown_by_default, true ); ?>
			/>

			<label for="_audioigniter_countdown_by_default">
				<?php esc_html_e( 'Track timer starts as coundown', 'audioigniter-pro' ); ?>
			</label>
		</div>
		<?php
	}

	public function action_metabox_settings_skip_amount( $object, $box ) {
		$skip_position = AudioIgniter()->get_post_meta( $object->ID, '_audioigniter_skip_amount', '' );
		?>
		<div class="ai-form-field">
			<label for="_audioigniter_skip_amount">
				<?php esc_html_e( 'Position skipping', 'audioigniter-pro' ); ?>
			</label>

			<input
				type="number"
				id="_audioigniter_skip_amount"
				class="ai-track-title"
				name="_audioigniter_skip_amount"
				placeholder="<?php esc_attr_e( 'Seconds to skip', 'audioigniter-pro' ); ?>"
				value="<?php echo esc_attr( $skip_position ); ?>"
			/>

			<p class="ai-field-help">
				<?php esc_html_e( "Set a number to allow skipping a track's position by that amount of seconds, or leave empty to disable.", 'audioigniter-pro' ); ?>
			</p>
		</div>
		<?php
	}

	public function action_metabox_settings_initial_track( $object, $box ) {
		$initial_track = AudioIgniter()->get_post_meta( $object->ID, '_audioigniter_initial_track', '' );
		?>
		<div class="ai-form-field">
			<label for="_audioigniter_initial_track">
				<?php esc_html_e( 'Starting Track', 'audioigniter-pro' ); ?>
			</label>

			<input
				type="number"
				id="_audioigniter_initial_track"
				class="ai-track-title"
				name="_audioigniter_initial_track"
				min="1"
				placeholder="<?php esc_attr_e( 'Starting track number', 'audioigniter-pro' ); ?>"
				value="<?php echo esc_attr( $initial_track ); ?>"
			/>

			<p class="ai-field-help">
				<?php esc_html_e( 'Starts playing from the track at this position. Leave empty to start from the beginning.', 'audioigniter-pro' ); ?>
			</p>
		</div>
		<?php
	}

	public function action_metabox_settings_tracks_delay( $object, $box ) {
		$tracks_delay = AudioIgniter()->get_post_meta( $object->ID, '_audioigniter_delay_between_tracks', '' );
		?>
		<div class="ai-form-field">
			<label for="_audioigniter_delay_between_tracks">
				<?php esc_html_e( 'Delay between tracks', 'audioigniter-pro' ); ?>
			</label>

			<input
				type="number"
				id="_audioigniter_delay_between_tracks"
				class="ai-track-title"
				name="_audioigniter_delay_between_tracks"
				min="0"
				placeholder="<?php esc_attr_e( 'Seconds to wait', 'audioigniter-pro' ); ?>"
				value="<?php echo esc_attr( $tracks_delay ); ?>"
			/>

			<p class="ai-field-help">
				<?php esc_html_e( 'Set a number of seconds to wait for before playing the next track.', 'audioigniter-pro' ); ?>
			</p>
		</div>
		<?php
	}

	public function action_metabox_settings_stop_on_track_finish( $object, $box ) {
		$stop_on_track_finish = AudioIgniter()->get_post_meta( $object->ID, '_audioigniter_stop_on_track_finish', 0 );
		?>
		<div class="ai-form-field">
			<input
				type="checkbox"
				class="ai-checkbox"
				id="_audioigniter_stop_on_track_finish"
				name="_audioigniter_stop_on_track_finish"
				value="1" <?php checked( $stop_on_track_finish, true ); ?>
			/>

			<label for="_audioigniter_stop_on_track_finish">
				<?php esc_html_e( 'Stop playing after each track finishes', 'audioigniter-pro' ); ?>
			</label>

			<p class="ai-field-help">
				<?php esc_html_e( 'This will override any repeat or cycling options that are set on the player.', 'audioigniter-pro' ); ?>
			</p>
		</div>
		<?php
	}

	public function action_metabox_settings_remember_last_player_position( $object, $box ) {
		$remember_last_player_position = AudioIgniter()->get_post_meta( $object->ID, '_audioigniter_remember_last_player_position', 0 );
		?>
		<div class="ai-form-field">
			<input
				type="checkbox"
				class="ai-checkbox"
				id="_audioigniter_remember_last_player_position"
				name="_audioigniter_remember_last_player_position"
				value="1" <?php checked( $remember_last_player_position, true ); ?>
			/>

			<label for="_audioigniter_remember_last_player_position">
				<?php esc_html_e( 'Remember last player position', 'audioigniter-pro' ); ?>
			</label>

			<p class="ai-field-help">
				<?php esc_html_e( 'This will force the player to start playing from the same spot for each user after they refresh or revisit the player.', 'audioigniter-pro' ); ?>
			</p>
		</div>
		<?php
	}

	public function filter_default_track_values( $values ) {
		$values['lyrics'] = '';

		return $values;
	}

	public function filter_playlist_endpoint_track( $track_response, $track, $playlist_id, $post ) {
		$track_response['lyrics'] = $track['lyrics'];

		return $track_response;
	}

	public function filter_get_playlist_data_attributes_array( $attrs, $post_id ) {
		$ai = AudioIgniter();

		$attrs['data-allow-track-loop']    = $ai->convert_bool_string( $ai->get_post_meta( $post_id, '_audioigniter_allow_track_loop', 0 ) );
		$attrs['data-shuffle']             = $ai->convert_bool_string( $ai->get_post_meta( $post_id, '_audioigniter_shuffle', 0 ) );
		$attrs['data-allow-playback-rate'] = $ai->convert_bool_string( $ai->get_post_meta( $post_id, '_audioigniter_allow_playback_rate', 0 ) );
		$attrs['data-timer-countdown']     = $ai->convert_bool_string( $ai->get_post_meta( $post_id, '_audioigniter_countdown_by_default', 0 ) );
		$attrs['data-skip-amount']         = $ai->get_post_meta( $post_id, '_audioigniter_skip_amount', '' );
		$attrs['data-initial-track']       = $ai->get_post_meta( $post_id, '_audioigniter_initial_track', '1' );
		$attrs['data-tracks-delay']        = $ai->get_post_meta( $post_id, '_audioigniter_delay_between_tracks', '' );
		$attrs['data-stop-on-finish']      = $ai->convert_bool_string( $ai->get_post_meta( $post_id, '_audioigniter_stop_on_track_finish', 0 ) );
		$attrs['data-remember-last']       = $ai->convert_bool_string( $ai->get_post_meta( $post_id, '_audioigniter_remember_last_player_position', 0 ) );

		$buttons = $ai->get_post_meta( $post_id, '_audioigniter_player_buttons', array() );

		$attrs['data-player-buttons'] = wp_json_encode( array_map( function ( $button ) {
			$button['icon'] = ! empty( $button['icon'] ) ? $this->icons->get_svg( $button['icon'] ) : '';

			return $button;
		}, $buttons ) );

		return $attrs;
	}

	public function action_save_post( $post_id ) {
		update_post_meta( $post_id, '_audioigniter_allow_track_loop', AudioIgniter()->sanitizer->checkbox_ref( $_POST['_audioigniter_allow_track_loop'] ) );
		update_post_meta( $post_id, '_audioigniter_allow_playback_rate', AudioIgniter()->sanitizer->checkbox_ref( $_POST['_audioigniter_allow_playback_rate'] ) );
		update_post_meta( $post_id, '_audioigniter_shuffle', AudioIgniter()->sanitizer->checkbox_ref( $_POST['_audioigniter_shuffle'] ) );
		update_post_meta( $post_id, '_audioigniter_countdown_by_default', AudioIgniter()->sanitizer->checkbox_ref( $_POST['_audioigniter_countdown_by_default'] ) );
		update_post_meta( $post_id, '_audioigniter_skip_amount', AudioIgniter()->sanitizer->intval_or_empty( $_POST['_audioigniter_skip_amount'] ) );
		update_post_meta( $post_id, '_audioigniter_initial_track', AudioIgniter()->sanitizer->intval_or_empty( $_POST['_audioigniter_initial_track'] ) );
		update_post_meta( $post_id, '_audioigniter_delay_between_tracks', AudioIgniter()->sanitizer->intval_or_empty( $_POST['_audioigniter_delay_between_tracks'] ) );
		update_post_meta( $post_id, '_audioigniter_stop_on_track_finish', AudioIgniter()->sanitizer->checkbox_ref( $_POST['_audioigniter_stop_on_track_finish'] ) );
		update_post_meta( $post_id, '_audioigniter_remember_last_player_position', AudioIgniter()->sanitizer->checkbox_ref( $_POST['_audioigniter_remember_last_player_position'] ) );

		update_post_meta( $post_id, '_audioigniter_player_buttons', $this->sanitize_metabox_player_buttons( $_POST['ai_player_buttons'], $post_id ) );
	}

	public function action_metabox_tracks_soundcloud_sync_button() {
		?>
		<button type="button" class="button ai-sync-soundcloud">
			<?php esc_html_e( 'Sync SoundCloud', 'audioigniter-pro' ); ?>
		</button>
		<?php
	}

	public function action_metabox_tracks_download_url_checkbox( $track, $uid ) {
		?>
		<label for="ai_playlist_tracks-<?php echo esc_attr( $uid ); ?>-download_uses_track_url" class="ai-form-field-checkbox-secondary">
			<input
				type="checkbox"
				class="ai-checkbox ai-track-download-uses-track-url"
				name="ai_playlist_tracks[<?php echo esc_attr( $uid ); ?>][download_uses_track_url]"
				id="ai_playlist_tracks-<?php echo esc_attr( $uid ); ?>-download_uses_track_url"
				value="1" <?php checked( (int) $track['download_uses_track_url'], 1 ); ?>
			/>

			<?php esc_html_e( 'Use the track URL as the download URL', 'audioigniter-pro' ); ?>
		</label>
		<?php
	}

	public function filter_shortcode_data_attributes_array( $data, $id, $post ) {
		$client_id = get_option( 'audioigniter_sc_client_id' );
		if ( ! empty( $client_id ) ) {
			$data['data-soundcloud-client-id'] = $client_id;
		}

		return $data;
	}

	/**
	 * Adds additional player types to AudioIgniter.
	 *
	 * @version 1.1.0
	 * @since   1.1.0
	 *
	 * @param array $player_types Player types array.
	 *
	 * @return array
	 */
	public function filter_player_types( $player_types ) {
		$player_types = array_merge( $player_types, array(
			'global-footer' => array(
				'label'      => __( 'Global Footer Player', 'audioigniter-pro' ),
				'no-support' => array(
					'show_credit',
					'max_width',
				),
				'info'       => __( "You've set this player as Global and it will appear at the bottom of your website, on every page. Please note that in the case that you've set multiple Global Players only the most recent one will be displayed.", 'audioigniter-pro' ),
			),
		) );

		return $player_types;
	}

	/**
	 * Determines if the [ai_playlist] shortcode should be short-circuited.
	 *
	 * @version 1.1.0
	 * @since   1.1.0
	 *
	 * @param bool    $shortcircuit Whether to short-circuit the shortcode.
	 * @param int     $id           The playlist ID.
	 * @param WP_Post $post         The playlist WP_Post object.
	 * @param array   $params       The playlist data attributes array.
	 *
	 * @return bool
	 */
	public function filter_shortcode_shortcircuit( $shortcircuit, $id, $post, $params ) {
		if ( ! empty( $params['data-player-type'] ) && 'global-footer' === $params['data-player-type'] ) {
			return true;
		}

		return $shortcircuit;
	}

	/**
	 * Return the post ID of the playlist that should be displayed in the global footer player.
	 *
	 * @version 1.1.0
	 * @since   1.1.0
	 *
	 * @return int|false
	 */
	public function get_global_footer_player_id() {

		$transient = get_transient( 'audioigniter_global_footer_player_id' );

		if ( false === $transient ) {
			$q = new WP_Query( array(
				'post_type'      => AudioIgniter()->post_type,
				'posts_per_page' => 1,
				'meta_query'     => array(
					array(
						'key'   => '_audioigniter_player_type',
						'value' => 'global-footer',
					),
				),
			) );

			$transient = array(
				'id' => false,
			);

			if ( $q->have_posts() ) {
				while ( $q->have_posts() ) {
					$q->the_post();
					$transient['id'] = get_the_ID();
					break;
				}
				wp_reset_postdata();
			}

			// Store array instead of id, as id may be false and we won't be able to distinguish from when the transient doesn't exist.
			// Non-expiring transient. It gets invalidated on post save.
			set_transient( 'audioigniter_global_footer_player_id', $transient );
		}

		return $transient['id'];
	}

	/**
	 * Echoes the output for the global footer player (most recent if multiples exist).
	 *
	 * @version 1.1.0
	 * @since   1.1.0
	 */
	public function global_footer_player() {
		$id = $this->get_global_footer_player_id();

		if ( empty( $id ) || ! AudioIgniter()->is_playlist( $id ) ) {
			return;
		}

		$post = get_post( $id );

		$params = apply_filters( 'audioigniter_global_footer_player_data_attributes_array', AudioIgniter()->get_playlist_data_attributes_array( $id ), $id, $post );
		$params = array_filter( $params, array( AudioIgniter()->sanitizer, 'array_filter_empty_null' ) );
		$params = AudioIgniter()->sanitizer->html_data_attributes_array( $params );

		// Returning a truthy value from the filter, will short-circuit execution of the global footer player.
		if ( false !== apply_filters( 'audioigniter_global_footer_player_shortcircuit', false, $id, $post, $params ) ) {
			return;
		}

		$data = '';
		foreach ( $params as $attribute => $value ) {
			$data .= sprintf( '%s="%s" ', sanitize_key( $attribute ), esc_attr( $value ) );
		}

		$output = sprintf( '<div id="audioigniter-%s" class="audioigniter-root" %s></div>',
			esc_attr( $id ),
			$data
		);

		echo $output; // phpcs:ignore WordPress.Security.EscapeOutput
	}

	/**
	 * Echoes the Player Buttons metabox markup.
	 *
	 * @since 1.7.0
	 *
	 * @param WP_Post $object
	 * @param array $box
	 */
	public function metabox_player_buttons( $object, $box ) {
		$buttons = AudioIgniter()->get_post_meta( $object->ID, '_audioigniter_player_buttons', array() );
		?>

		<div class="ai-container">
			<?php $this->metabox_player_buttons_field_controls( 'top' ); ?>

			<div class="ai-fields-container ai-fields-container-player-buttons ai-fields-sortable">
				<?php
					if ( ! empty( $buttons ) ) {
						foreach ( $buttons as $button ) {
							$this->metabox_player_buttons_repeatable_field( $button );
						}
					} else {
						$this->metabox_player_buttons_repeatable_field();
					}
				?>
			</div>

			<?php $this->metabox_player_buttons_field_controls( 'bottom' ); ?>
		</div>
		<?php
	}

	/**
	 * Outputs the field controls for the Player Buttons metabox markup.
	 *
	 * @since 1.7.0
	 *
	 * @param string $location
	 */
	protected function metabox_player_buttons_field_controls( $location ) {
		?>
		<div class="ai-field-controls-wrap">
			<div class="ai-field-controls">
				<button type="button" class="button ai-add-field ai-add-field-<?php echo esc_attr( $location ); ?>">
					<span class="dashicons dashicons-plus-alt"></span>
					<?php esc_html_e( 'Add Button', 'audioigniter-pro' ); ?>
				</button>
			</div>

			<div class="ai-field-controls-visibility">
				<a href="#" class="ai-fields-expand-all">
					<?php esc_html_e( 'Expand All', 'audioigniter-pro' ); ?>
				</a>
				<a href="#" class="ai-fields-collapse-all">
					<?php esc_html_e( 'Collapse All', 'audioigniter-pro' ); ?>
				</a>
			</div>
		</div>
		<?php
	}

	/**
	 * Outputs a single repeatable Player Button.
	 *
	 * @since 1.7.0
	 *
	 * @param array $button
	 */
	protected function metabox_player_buttons_repeatable_field( $button = array() ) {
		$button = wp_parse_args( $button, $this->get_default_repeatable_player_button_values() );

		$icon  = $button['icon'];
		$title = $button['title'];
		$url   = $button['url'];

		$uid = uniqid();
		?>
		<div class="ai-field-repeatable" data-uid="<?php echo esc_attr( $uid ); ?>">
			<div class="ai-field-head">

				<span class="ai-field-sort-handle">
					<span class="screen-reader-text"><?php esc_html_e( 'Move button', 'audioigniter-pro' ); ?></span>
					<span class="dashicons dashicons-move"></span>
				</span>

				<span class="ai-field-title"><?php echo wp_kses( $title, array() ); ?></span>

				<button type="button" class="ai-field-toggle button-link">
					<span class="screen-reader-text">
						<?php esc_html_e( 'Toggle button visibility', 'audioigniter-pro' ); ?>
					</span>
					<span class="toggle-indicator"></span>
				</button>
			</div>

			<div class="ai-field-container ai-field-container-links">

				<div class="ai-field-split">
					<div class="ai-form-field">
						<label
							for="ai_player_buttons-<?php echo esc_attr( $uid ); ?>-title"
							class="screen-reader-text">
							<?php esc_html_e( 'Title', 'audioigniter-pro' ); ?>
						</label>
						<input
							type="text"
							id="ai_player_buttons-<?php echo esc_attr( $uid ); ?>-title"
							class="ai-track-title"
							name="ai_player_buttons[<?php echo esc_attr( $uid ); ?>][title]"
							placeholder="<?php esc_attr_e( 'Title', 'audioigniter-pro' ); ?>"
							value="<?php echo esc_attr( $title ); ?>"
						/>
					</div>

					<div class="ai-form-field">
						<label
							for="ai_player_buttons-<?php echo esc_attr( $uid ); ?>-icon"
							class="screen-reader-text">
							<?php esc_html_e( 'Icon', 'audioigniter-pro' ); ?>
						</label>
						<select
							id="ai_player_buttons-<?php echo esc_attr( $uid ); ?>-icon"
							class="ai-player-button-icon"
							name="ai_player_buttons[<?php echo esc_attr( $uid ); ?>][icon]"
						>
							<option value=""><?php esc_html_e( 'Icon', 'audioigniter-pro' ); ?></option>
							<?php echo $this->icons->get_dropdown_options( $icon ); ?>
						</select>
					</div>
				</div>

				<div class="ai-field-split">
					<div class="ai-form-field">
						<label
							for="ai_player_buttons-<?php echo esc_attr( $uid ); ?>-url"
							class="screen-reader-text">
							<?php esc_html_e( 'Link', 'audioigniter-pro' ); ?>
						</label>
						<input
							type="text"
							id="ai_player_buttons-<?php echo esc_attr( $uid ); ?>-url"
							class="ai-player-button-url"
							name="ai_player_buttons[<?php echo esc_attr( $uid ); ?>][url]"
							placeholder="<?php esc_attr_e( 'Link', 'audioigniter-pro' ); ?>"
							value="<?php echo esc_url( $url ); ?>"
						/>
					</div>

					<button type="button" class="button ai-remove-field">
						<span class="dashicons dashicons-dismiss"></span>
						<?php esc_html_e( 'Remove Button', 'audioigniter-pro' ); ?>
					</button>
				</div>

			</div>
		</div>
		<?php
	}

	/**
	 * Returns the default values of a single repeatable Player Button/
	 *
	 * @since 1.7.0
	 */
	private function get_default_repeatable_player_button_values() {
		return apply_filters( 'audioigniter_pro_default_player_button_values', array(
			'icon'  => '',
			'title' => '',
			'url'   => '',
		) );
	}

	/**
	 * Sanitizes a single repeatable Player Button.
	 *
	 * @since 1.7.0
	 *
	 * @param array    $post_buttons Input values to sanitize, as passed by the player buttons metabox.
	 * @param int|null $post_id      Optional. Post ID where the buttons belongs to.
	 *
	 * @return array
	 */
	private function sanitize_metabox_player_buttons( $post_buttons, $post_id = null ) {
		if ( empty( $post_buttons ) || ! is_array( $post_buttons ) ) {
			return array();
		}

		$buttons = array();

		foreach ( $post_buttons as $uid => $button_data ) {
			$button = $this->sanitize_repeating_player_button( $button_data, $post_id, $uid );
			if ( false !== $button ) {
				$buttons[] = $button;
			}
		}

		return apply_filters( 'audioigniter_sanitize_player_buttons', $buttons, $post_buttons, $post_id );
	}

	/**
	 * Sanitizes a single repeatable player button.
	 *
	 * @since 1.7.0

	 * @uses AudioIgniter_Pro::get_default_repeatable_player_button_values()
	 *
	 * @param array    $button     Button array to sanitize.
	 * @param int|null $post_id    Optional. Post ID where the button belongs to.
	 * @param string   $button_uid Optional. UID that identifies the button in the metabox list.
	 *
	 * @return array|false Array if at least one field is completed, false otherwise.
	 */
	private function sanitize_repeating_player_button( $button, $post_id = null, $button_uid = '' ) {
		$button = wp_parse_args( $button, $this->get_default_repeatable_player_button_values() );

		$sanitized_button = array();

		$sanitized_button['title'] = sanitize_text_field( $button['title'] );
		$sanitized_button['icon']  = sanitize_text_field( $button['icon'] );
		$sanitized_button['url']   = esc_url_raw( $button['url'] );

		$sanitized_button = array_map( 'trim', $sanitized_button );

		$tmp = array_filter( $sanitized_button );
		if ( empty( $tmp ) ) {
			$sanitized_button = false;
		}

		return apply_filters( 'audioigniter_sanitize_repeating_player_button', $sanitized_button, $button, $post_id, $button_uid );
	}

	public function fiter_register_post_type_args( $args, $post_type ) {
		if ( AudioIgniter()->post_type !== $post_type ) {
			return $args;
		}

		$args['show_in_rest'] = true;

		return $args;
	}

	public function invalidate_transient_global_footer_player_id( $post_id ) {
		delete_transient( 'audioigniter_global_footer_player_id' );
	}


	public static function notice_plugin_required() {
		?>
		<div class="error notice">
			<p>
				<?php echo wp_kses(
					sprintf(
					/* translators: %s is a URL. */
						__( 'AudioIgniter Pro requires AudioIgniter. Please install and activate the free <a href="%s" target="_blank">AudioIgniter</a> plugin.', 'audioigniter-pro' ),
						'https://wordpress.org/plugins/audioigniter/'
					),
					array(
						'a' => array(
							'href'   => true,
							'target' => true,
						),
					)
				); ?>
			</p>
		</div>
		<?php
	}

	public function plugin_activated() {
		if ( ! current_user_can( 'activate_plugins' ) ) {
			return;
		}

		if ( class_exists( 'AudioIgniter' ) ) {
			$this->register_taxonomies();
		}

		flush_rewrite_rules();
	}

	public function plugin_deactivated() {
		if ( ! current_user_can( 'activate_plugins' ) ) {
			return;
		}

		unregister_taxonomy( 'ai_playlist_category' );

		flush_rewrite_rules();
	}

	public static function plugin_basename() {
		return plugin_basename( __FILE__ );
	}

	public function plugin_url() {
		return self::$plugin_url;
	}

	public function plugin_path() {
		return self::$plugin_path;
	}
}

/**
 * Main instance of AudioIgniter_Pro.
 *
 * Returns the working instance of AudioIgniter_Pro. No need for globals.
 *
 * @since  1.0.0
 * @return AudioIgniter_Pro
 */
function AudioIgniter_Pro() {
	return AudioIgniter_Pro::instance();
}

// Bootstrap before AudioIgniter (priority 10) as we need to hook onto actions before they run.
add_action( 'plugins_loaded', array( AudioIgniter_Pro(), 'plugin_setup' ), 8 );
register_activation_hook( __FILE__, array( AudioIgniter_Pro(), 'plugin_activated' ) );
register_deactivation_hook( __FILE__, array( AudioIgniter_Pro(), 'plugin_deactivated' ) );
