<?php
/**
 * Class CrispSettings
 *
 * @package Crisp
 */

namespace Crisp;

/**
 * Class CrispSettings
 *
 * @package Crisp
 */
class CrispSettings {
	/**
	 * Instance of CrispSettings.
	 *
	 * @var CrispSettings|null
	 */
	private static $instance = null;

	/**
	 * Init admin settings
	 */
	public function __construct() {
		$this->init();
	}

	/**
	 * Registers admin settings.
	 */
	public function init() {
		add_action('admin_menu', array( $this, 'register_admin_menu' ));
		add_action('admin_enqueue_scripts', array( $this, 'register_crisp_admin_style'));
		add_action('admin_notices', array($this, 'register_crisp_plugin_notice'));
		add_action('plugins_loaded', array($this, 'register_crisp_plugin_textdomain') );
		add_action('admin_init', array($this, 'register_crisp_plugin_settings'));
		add_action('admin_init', array($this, 'register_crisp_plugin_onboarding'));
	}

	/**
	 * Registers plugin menu in WP menu bar.
	 */
	public function register_admin_menu() {
		add_menu_page(
			__('Crisp Settings', 'crisp-woocommerce'),
			__('Crisp Settings', 'crisp-woocommerce'),
			'administrator',
			'crisp',
			array( $this, 'crisp_settings_page' ),
			plugins_url('../assets/images/icon-favicon.svg', __FILE__)
		);
	}

	/**
	 * Registers Crisp settings CSS
	 */
	public function register_crisp_admin_style() {
		wp_enqueue_style('admin_crisp_style', plugins_url('../assets/stylesheets/style.css', __FILE__), array(), '1.0');
	}

	/**
	 * Registers plugin notice
	 */
	public function register_crisp_plugin_notice() {
		$website_id = get_option('website_id');

		if (empty($website_id)) {
			$admin_url = admin_url('admin.php?page=crisp');
			?>
				<div class='notice notice-warning is-dismissible notice-crisp'>
					<p>
						<img src='<?php echo esc_url(plugins_url('../assets/images/icon-favicon.svg', __FILE__)); ?>' height='16' style='margin-bottom: -3px' />
						&nbsp;
						<?php
							echo sprintf(
								/* translators: %1s: opening of anchor tag. %2s: closing of anchor tag */
								esc_html(__('The Crisp plugin isn’t connected right now. To display Crisp widget on your WordPress site, %1$sconnect the plugin now%2$s. The configuration only takes 1 minute!', 'crisp-woocommerce') ),
								"<a href='$admin_url'>",
								'</a>'
							);
						?>
					</p>
				</div>
			<?php
		}
	}

	/**
	 * Registers the plugin textdomain
	 */
	public function register_crisp_plugin_textdomain() {
		load_plugin_textdomain('crisp-woocommerce', false, basename( dirname( __FILE__ ) ) . '/languages/' );
	}

	/**
	 * Registers the plugin settings
	 */
	public function register_crisp_plugin_settings() {
		register_setting( 'crisp-plugin-settings-group', 'website_id' );
		add_option('crisp_onboarding', false);
	}

	/**
	 * Registers whether the customer needs to be onboarded.
	 */
	public function register_crisp_plugin_onboarding() {
		global $wp_rewrite;
		$onboarding = get_option('crisp_onboarding');
		$website_id = get_option('website_id');

		if ( empty($website_id) && ( empty($onboarding) || !$onboarding ) ) {
			update_option('crisp_onboarding', true);
			wp_redirect(admin_url('admin.php?page=crisp'));
		}

		// If website is not using permalinks, set permalink structure as \
		// it is required to use the Woocommerce REST API
		if (!$wp_rewrite->using_permalinks()) {
			$wp_rewrite->set_permalink_structure('/%post_id%/');

			$wp_rewrite->flush_rules();
		}
	}

	/**
	 * Creates settings page.
	 */
	public function crisp_settings_page() {
		if (current_user_can('administrator')) {
			if ( isset($_GET['_wpnonce']) && wp_verify_nonce(sanitize_key($_GET['_wpnonce']))) {
				if (isset($_GET['crisp_website_id']) && !empty($_GET['crisp_website_id'])) {
					update_option('website_id', sanitize_text_field($_GET['crisp_website_id']));

					// Clear WP Rocket Cache if needed
					if (function_exists('rocket_clean_domain')) {
						rocket_clean_domain();
					}

					// Clear WP Super Cache if needed
					if (function_exists('wp_cache_clean_cache')) {
						global $file_prefix;
						wp_cache_clean_cache($file_prefix, true);
					}
				}

				if (isset($_GET['crisp_verify']) && !empty($_GET['crisp_verify'])) {
					update_option('website_verify', sanitize_key($_GET['crisp_verify']));
				}
			}
		}

		if (isset( $_SERVER['SERVER_PORT']) && isset($_SERVER['HTTP_HOST']) && isset( $_SERVER['REQUEST_URI'] ) ) {
			$wp_nonce = wp_create_nonce();

			$http_callback = 'http' . ( ( 443 == $_SERVER['SERVER_PORT'] ) ? 's://' : '://' ) . sanitize_text_field($_SERVER['HTTP_HOST']) . sanitize_text_field($_SERVER['REQUEST_URI']) . '&user_id=$admin_id&_wpnonce=' . $wp_nonce;
		}

		$website_id = get_option('website_id');
		$is_crisp_working = isset($website_id) && !empty($website_id);
		$admin_id = wp_get_current_user()->ID;
		$http_callback = urlencode($http_callback);
		$admin_email = wp_get_current_user()->user_email;
		$admin_name = wp_get_current_user()->display_name;
		$website_name = get_bloginfo('name');
		$website_domain = get_site_url();
		$website_domain = str_replace('http://', '', $website_domain);
		$website_domain = str_replace('https://', '', $website_domain);
		$website_domain = preg_replace('(:[0-9]{1,6})', '', $website_domain);
		$crisp_plugin_id = CRISP_PLUGIN_ID;

		$add_to_crisp_link = esc_url_raw('https://app.crisp.chat/initiate/plugin/' . $crisp_plugin_id . '?payload=' . $http_callback . '&user_email=' . $admin_email . '&user_name=' . $admin_name . '&website_name=' . $website_name . '&website_domain=' . $website_domain);

		if ($is_crisp_working) {
			include_once(plugin_dir_path( __FILE__ ) . '../views/settings_installed.php');
		} else {
			include_once(plugin_dir_path( __FILE__ ) . '../views/settings_install.php');
		}
	}

	/**
	 * Returns instance of CrispSettings (singleton pattern).
	 *
	 * @return CrispSettings|null
	 */
	public static function get_instance() {
		if ( !isset( static::$instance ) ) {
			static::$instance = new static();
		}

		return static::$instance;
	}
}
