<?php
/**
 * Class CrispWidget
 *
 * @package Crisp
 */

namespace Crisp;

/**
 * Class CrispWidget
 *
 * @package Crisp
 */
class CrispWidget {
	/**
	 * Instance of CrispWidget.
	 *
	 * @var CrispWidget|null
	 */
	private static $instance = null;

	/**
	 * CrispWidget constructor
	 */
	public function __construct() {
		$this->init();
	}

	/**
	 * Init Crisp Wigdet
	 */
	public function init() {
		add_action('wp_enqueue_scripts', array($this, 'crisp_enqueue_script'));
		add_action('script_loader_tag', array($this, 'crisp_enqueue_async'), 10, 2);
	}

	/**
	 * Enqueue Scripts
	 */
	public function crisp_enqueue_script() {
		$website_id = get_option('website_id');
		$locale = str_replace('_', '-', strtolower(get_locale()));
		$locale = preg_replace('/([a-z]{2}-[a-z]{2})(-.*)/', '$1', $locale);

		if (!isset($website_id) || empty($website_id)) {
			return;
		}
		
	 $output="
			window.\$crisp=[];
			if (!window.CRISP_RUNTIME_CONFIG) {
				window.CRISP_RUNTIME_CONFIG = {
					locale : '$locale'
				}
			} else if (!window.CRISP_RUNTIME_CONFIG.locale) {
				window.CRISP_RUNTIME_CONFIG.locale = '$locale'
			}
			CRISP_WEBSITE_ID = '$website_id';";

		$output .= $this->crisp_sync_wordpress_user();
		$output .= $this->prepare_post_visitor_data();

		wp_enqueue_script('crisp', 'https://client.crisp.chat/l.js', array(), '1.0', true);
		wp_add_inline_script('crisp', $output, 'before');
	}

	/**
	 * Enqueue Async script
	 */
	public function crisp_enqueue_async( $tag, $handle) {
		if ('crisp' !== $handle ) {
			return $tag;
		}

		return str_replace('src', ' async src', $tag );
	}

	public function crisp_sync_wordpress_user() {
		$output = "\$crisp.push(['set', 'session:segments', [['woocommerce']]]);";

		if (is_user_logged_in()) {
			$current_user = wp_get_current_user();
		}

		if (!isset($current_user)) {
			return '';
		}

		$website_verify = get_option('website_verify');

		$email = $current_user->user_email;
		$nickname = $current_user->display_name;

		if (!empty($email) && empty($website_verify)) {
			$output .= "\$crisp.push(['set', 'user:email', '" . $email . "']);";
		} else if (!empty($email)) {
			$hmac = hash_hmac('sha256', $email, $website_verify);
			$output .= "\$crisp.push(['set', 'user:email', ['" . $email . "', '" . $hmac . "']]);";
		}

		if (!empty($nickname)) {
			$output .= "\$crisp.push(['set', 'user:nickname', '" . $nickname . "']);";
		}

		return $output;
	}

	/**
	 * Post customer to Crisp
	 */
	public function post_customer() {
		if (WC()->customer == null) {
			// Return if there is no customer
			return '';
		}

		if (WC()->customer->get_email() == null) {
			return '';
		}

		$customer_data = array(
			'id' => WC()->customer->get_id(),
			'email' =>  WC()->customer->get_email()
		);

		$body = wp_json_encode($customer_data);
		$website_id = get_option('website_id');
		$plugin_url = CRISP_PLUGIN_URL;

		$output = "
			fetch('$plugin_url/visitors/website/$website_id/session/' + _sessionID + '/customer', {
				method: 'POST',
				headers: {
					'Content-Type': 'application/json',
					'Accept': 'application/json'
				},
				body: JSON.stringify($body)
			});";

		return $output;
	}

	/**
	 * Post Cart to Crisp
	 */
	public function post_cart() {
		$cart_data = (object) array();

		if (WC()->cart != null && !WC()->cart->is_empty()) {
			$cart =  WC()->cart->get_cart();
			$cart_items = array();

			foreach ( $cart as $cart_item_key => $cart_item ) {
				$product = $cart_item['data'];

				$item = (object) $product->get_data();

				$product_image = wp_get_attachment_image_src($product->get_image_id());
				$permalink = $product->get_permalink();
				$quantity = $cart_item['quantity'];

				if ($product_image) {
					$image = (object) array(
						'id' => $product->get_image_id(),
						'source' => $product_image[0]
					);

					$item->image = $image;
				}

				if ($permalink) {
					$item->permalink = $permalink;
				}

				if ($quantity) {
					$item->quantity = $quantity;
				}

				array_push($cart_items, $item);
			}

			$cart_data = (object) array(
				'base_url' => get_site_url(),
				'totals' => WC()->cart->get_totals(),
				'currency' => get_woocommerce_currency(),
				'items' => $cart_items,
			);
		}

		$body = wp_json_encode($cart_data);

		$headers = array(
			'Content-Type' => 'application/json',
		);

		$args = array(
			'body' => $body,
			'headers' => $headers
		);

		$website_id = get_option('website_id');
		$plugin_url = CRISP_PLUGIN_URL;

		$output = "
			fetch('$plugin_url/visitors/website/$website_id/session/' + _sessionID + '/cart', {
				method: 'POST',
				headers: {
					'Content-Type': 'application/json',
					'Accept': 'application/json'
				},
				body: JSON.stringify($body)
			});";

		return $output;
	}

	/**
	 * Prepare Data to post to Crisp
	 */
	public function prepare_post_visitor_data() {
		$output = "window.CRISP_READY_TRIGGER = function () {
			let _sessionID = \$crisp.get('session:identifier');";

		if (function_exists( 'WC' )) {
			$output .= $this->post_cart();
			$output .= $this->post_customer();
		};

		$output .= '};';

		return $output;
	}

	/**
	 * Returns instance of CrispWidget (singleton pattern).
	 *
	 * @return CrispWidget|null
	 */
	public static function get_instance() {
		if ( !isset( static::$instance ) ) {
			static::$instance = new static();
		}

		return static::$instance;
	}
}
