<?php
/*
 * Table Rate Shipping Method Extender Class
 */

if ( ! defined( 'ABSPATH' ) )
	exit;

// Check if WooCommerce is active
if ( class_exists( 'WooCommerce' ) && function_exists('dokan') ) {

	if ( class_exists( 'BETRS_Dokan' ) ) return;

	class BETRS_Dokan {

		/**
		 * Cloning is forbidden. Will deactivate prior 'instances' users are running
		 *
		 * @since 4.0
		 */
		public function __clone() {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Cloning this class could cause catastrophic disasters!', 'be-table-ship' ), '4.0' );
		}

		/**
		 * Unserializing instances of this class is forbidden.
		 *
		 * @since 4.0
		 */
		public function __wakeup() {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Unserializing is forbidden!', 'be-table-ship' ), '4.0' );
		}

		/**
		 * __construct function.
		 *
		 * @access public
		 * @return void
		 */
		function __construct() {
			global $betrs_shipping;

			// CSS includes
			$ver = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? null : sanitize_text_field( $betrs_shipping->get_version() );
			wp_enqueue_style( 'betrs-dashboard', plugins_url( 'assets/css/dashboard.css', __DIR__ ), false, $ver );

			// manage dokan on the frontend vendor dashboard
			add_filter( 'dokan_get_dashboard_settings_nav', array( $this, 'add_vendor_dashboard_link' ), 23, 1 );
			add_action( 'dokan_render_settings_content', array( $this, 'add_shipping_settings' ), 23 );
			add_filter( 'betrs_user_management_method_link', array( $this, 'modify_frontend_link' ), 10, 4 );
			add_filter( 'betrs_user_management_method_back_link', array( $this, 'modify_frontend_back_link' ), 10, 3 );
			add_action( 'betrs_user_management_display', array( $this, 'section_table_rates' ), 10, 1 );

			// add ability to narrow method by vendor
			add_filter( 'betrs_user_modification_types', array( $this, 'add_user_modification' ), 10, 1 );
			add_filter( 'woocommerce_shipping_instance_form_fields_betrs_shipping', array( $this, 'add_user_modification_settings' ), 10, 1 );

			// add 'Dokan' field to conditions list
			add_filter( 'betrs_shipping_cost_conditionals', array( $this, 'add_vendor_condition' ), 10, 1 );
			add_filter( 'betrs_shipping_cost_conditionals_secondary', array( $this, 'add_vendor_condition_secondary' ), 10, 1 );
			add_action( 'betrs_shipping_cost_conditionals_tertiary', array( $this, 'add_vendor_condition_tertiary' ), 10, 5 );
			add_filter( 'betrs_calculated_totals-per_order', array( $this, 'get_vendor_data_order' ), 10, 2 );
			add_filter( 'betrs_calculated_totals-per_item', array( $this, 'get_vendor_data_item' ), 10, 2 );
			add_filter( 'betrs_calculated_totals-per_class', array( $this, 'get_vendor_data_class' ), 10, 2 );
			add_filter( 'betrs_determine_condition_result', array( $this, 'compare_vendor_condition' ), 10, 3 );
		}


		/**
		 * add Table Rate to list of enabled shipping options.
		 *
		 * @access public
		 * @param array $package (default: array())
		 * @return array
		 */
		function add_vendor_dashboard_link( $settings_sub ) {
			// setup new menu item
			$new_menu_item = array(
				'title' 	=> 'Table Rate Shipping',
				'icon' 		=> '<i class="fa fa-th-large"></i>',
				'url' 		=> dokan_get_navigation_url( 'settings/betrs' ),
				'pos' 		=> 75,
				'permission'=> 'dokan_view_store_shipping_menu',
		      );

			// add menu item to navigation
			$settings_sub['betrs'] = $new_menu_item;

			return $settings_sub;
		}


		/**
		 * add Table Rate to list of enabled shipping options.
		 *
		 * @access public
		 * @param array $package (default: array())
		 * @return array
		 */
		function add_shipping_settings() {
			global $wp, $betrs_shipping, $allowedposttags;

        	// determine page variables
        	$request = $wp->request;
        	$active = explode( '/', $request );

			// only show on 'Shipping' tab of dashboard settings
			$tab = $active[2];
        	if( ! $tab || sanitize_title( $tab ) !== 'betrs' ) return;

        	// determine page content type
        	if( ( isset( $active[3] ) && sanitize_key( $active[3] ) === 'instance-id' ) && intval( $active[4] ) ) {
        		// include necessary files for WP List Table class
        		$this->add_frontend_includes();

        		// display individual method settings page
        		$betrs_shipping->user_management->display_management_instance( intval( $active[4] ), 'dokan' );
        	} else {
        		// display list of zones and shipping methods
		    	printf( '<h1 class="entry-title">%s</h1>', esc_html__( 'Table Rate Shipping Methods', 'be-table-ship' ) );
		    	$betrs_shipping->user_management->display_management_home( 'dokan' );
		    }
		}


		/**
		 * change the URL for the single shipping method settings page.
		 *
		 * @access public
		 * @param array $package (default: array())
		 * @return array
		 */
		function modify_frontend_link( $link, $template, $instance_id, $method ) {
			// only modify Dokan returns
			if( $template !== 'dokan' ) return $link;

			// only run on frontend editor
			if( is_admin() ) return $link;

			$link = dokan_get_navigation_url( 'settings/betrs/instance-id/' . esc_attr( $instance_id ) );

			return $link;
		}


		/**
		 * change the URL for the 'Back to Shipping Zones' link.
		 *
		 * @access public
		 * @param array $package (default: array())
		 * @return array
		 */
		function modify_frontend_back_link( $link, $template, $instance_id ) {
			// only modify Dokan returns
			if( $template !== 'dokan' ) return $link;

			// only run on frontend editor
			if( is_admin() ) return $link;

			$link = dokan_get_navigation_url( 'settings/betrs' );

			return $link;
		}


		/**
		 * print out table's configuration.
		 *
		 * @access public
		 */
		function section_table_rates( $shipping_method ) {
			global $betrs_shipping;

			// get saved settings
			$settings_query = get_option( $shipping_method->get_options_save_name() );
			$settings = ( is_array( $settings_query ) && isset( $settings_query['settings'] ) ) ? $settings_query['settings'] : array();

			// gather columns for table
	        $columns = apply_filters( 'betrs_shipping_table_columns', array(
	            'cb'            => '&nbsp;',
	            'conditions'    => esc_html__( 'Conditions', 'be-table-ship' ),
	            'costs'         => esc_html__( 'Costs', 'be-table-ship' ),
	            'description'   => esc_html__( 'Description', 'be-table-ship' ) . '<span class="woocommerce-help-tip" data-tip="' . esc_html__( 'Can override or add on to option\'s description (optional)' ) . '"></span>',
	            'sort'          => '',
	            )
	        );
	        if( isset( $columns['cb'] ) ) unset( $columns['cb'] );
	        if( isset( $columns['sort'] ) ) unset( $columns['sort'] );

	        include( plugin_dir_path( __FILE__ ) . 'views/dokan.table-rates.php' );
		}


		/**
		 * add dashboard includes to frontend page.
		 *
		 * @access public
		 * @param array $package (default: array())
		 * @return array
		 */
		function add_frontend_includes() {
			global $hook_suffix;
			
			$hook_suffix = '';

			// help operate in frontend
			if( ! is_admin() ) {
				require_once( ABSPATH . 'wp-admin/includes/screen.php' );
				require_once( ABSPATH . 'wp-admin/includes/class-wp-screen.php' );
				require_once( ABSPATH . 'wp-admin/includes/template.php' );
			}
		}


		/**
		 * add_user_modification function.
		 *
		 * @access public
		 * @param array $package (default: array())
		 * @return array
		 */
		function add_user_modification( $user_types ) {
			$user_types['dokan-vendors'] = esc_html__( 'Specific Vendors', 'be-table-ship' );

			return $user_types;
		}


		/**
		 * add_user_modification_settings function.
		 *
		 * @access public
		 * @param array $package (default: array())
		 * @return array
		 */
		function add_user_modification_settings( $settings ) {
			// check for vendors setup before proceeding
			$dokan = dokan();
			if( ! isset( $dokan ) || ! isset( $dokan->vendors ) )
				return $settings;

			// get Dokan vendors
			$get_vendors = dokan_get_sellers( array( 'number' => -1 ) );
			$vendors = $get_vendors['users'];
			$vendors_ar = array();
			foreach( $vendors as $key => $vendor ) {
				$vendors_ar[ $vendor->data->ID ] = $vendor->data->display_name;
			}

			$settings['user_permissions']['settings']['user_modification_dokan'] = array(
				'title' 			=> esc_html__( 'Vendors', 'be-table-ship' ),
				'type' 				=> 'multiselect',
				'class'         	=> 'wc-enhanced-select',
				'default' 			=> '',
				'options' 			=> $vendor_ar,
			);

			return $settings;
		}


		/**
		 * add_vendor_condition function.
		 *
		 * @access public
		 * @param array $package (default: array())
		 * @return array
		 */
		function add_vendor_condition( $conditions ) {
			global $current_screen;

			if( isset( $current_screen ) && isset( $current_screen->parent_base ) && $current_screen->parent_base == 'betrs-manage-shipping' ) {
				return $conditions;
			}

		    // add new option to list
		    $conditions['dokan_vendor'] = 'Dokan Vendor';
		    
		    return $conditions;
		}


		/**
		 * add_vendor_condition_secondary function.
		 *
		 * @access public
		 * @param array $package (default: array())
		 * @return array
		 */
		function add_vendor_condition_secondary( $conditions ) {
		    // add new option to list
		    $conditions['includes']['conditions'][] = 'dokan_vendor';
		    $conditions['excludes']['conditions'][] = 'dokan_vendor';
		    
		    return $conditions;
		}


		/**
		 * add_vendor_condition_tertiary function.
		 *
		 * @access public
		 * @param string $cond_type, array $item, int $row_ID, int $option_ID (default: null), int $cond_key (default: 0)
		 * @return null
		 */
		function add_vendor_condition_tertiary( $cond_type, $item, $row_ID, $option_ID = null, $cond_key = 0 ) {
			// sanitize inputs
			$cond_tertiary = array();
			$cond_type = sanitize_title( $cond_type );
			$row_ID = (int) $row_ID;
			if( isset( $item['cond_tertiary'] ) && is_array( $item['cond_tertiary'] ) ) {
				$cond_tertiary = array_map( 'intval', $item['cond_tertiary'] );
			}

	        if( isset( $option_ID ) ) {
	            // setup table rate form fields
	            $option_ID = (int) $option_ID;
	            $op_name_tertiary = "cond_tertiary[" . $option_ID . "][" . $row_ID . "][" . $cond_key . "]";
	        } else {
	            // setup method condition form fields
	            $op_name_tertiary = "method_cond_tertiary[" . $row_ID . "]";
	        }

			// check if type is for Dokan
			if( sanitize_title( $cond_type ) != 'dokan_vendor' ) return;

			// get Dokan vendors
			$get_vendors = dokan_get_sellers( array( 'number' => -1 ) );
			$vendors = $get_vendors['users'];
?>
<select name="<?php echo sanitize_text_field( $op_name_tertiary ); ?>[]" class="cond_tertiary wc-enhanced-select cond_multiple" multiple="multiple">
        <?php
            // decode if passed through importer
            if( is_string( $cond_tertiary ) )
                $cond_tertiary = json_decode( sanitize_text_field( $cond_tertiary ) );

            $sel_vendors = ( is_array( $cond_tertiary ) ) ? $cond_tertiary : array();
            foreach ( $vendors as $vendor ) {
            	$vendor_data = get_userdata( $vendor->ID );
                echo '<option value="' . esc_attr( $vendor->ID ) . '"' . selected( in_array( $vendor->ID, $sel_vendors ), true, false ) . '>' . wp_kses_post( $vendor_data->display_name ) . '</option>';
            }
        ?>
</select>
<?php
			return;
		}


		/**
		 * get_vendor_data_order function.
		 *
		 * @access public
		 * @param array $data, array $items
		 * @return array
		 */
		function get_vendor_data_order( $data, $items ) {
			$vendor_ids = array();

			// cycle through products
			if( is_array( $items ) && ! empty( $items ) ) {
				foreach( $items as $key => $item ) {
					$vendor = dokan_get_vendor_by_product( $item['data'] );
					$vendor_id = $vendor->id;

					if( is_array( $vendor_id ) ) {
						$merged = array_merge( $vendor_ids, $vendor_id );
						$vendor_ids = array_unique( $merged );
					} else {
						$vendor_ids[] = (int) $vendor_id;
					}
				}
				$data['dokan_vendors'] = $vendor_ids;
			}

			return $data;
		}


		/**
		 * get_vendor_data_item function.
		 *
		 * @access public
		 * @param array $data, array $items
		 * @return array
		 */
		function get_vendor_data_item( $data, $items ) {
			// cycle through products
			if( is_array( $data ) && ! empty( $data ) ) {
				foreach( $data as $key => $item ) {
					$vendor = dokan_get_vendor_by_product( $key );
					$vendor_id = $vendor->id;

					if( is_array( $vendor_id ) ) {
						$data[ $key ]['dokan_vendors'] = $vendor_id;
					} else {
						$data[ $key ]['dokan_vendors'] = array( intval( $vendor_id ) );
					}
				}
			}

			return $data;
		}


		/**
		 * get_vendor_data_class function.
		 *
		 * @access public
		 * @param array $data, array $items
		 * @return array
		 */
		function get_vendor_data_class( $data, $items ) {
			// cycle through products
			if( is_array( $data ) && ! empty( $data ) ) {
				foreach( $data as $key => $item ) {
					$vendor_ids = array();
					foreach( $item['products'] as $pid ) {
						$vendor = dokan_get_vendor_by_product( $pid );
						$vendor_id = $vendor->id;

						if( is_array( $vendor_id ) ) {
							$merged = array_merge( $vendor_ids, $vendor_id );
							$vendor_ids = array_unique( $merged );
						} else {
							$vendor_ids[] = (int) $vendor_id;
						}
					}
					
					$data[ $key ]['dokan_vendors'] = $vendor_ids;
				}
			}

			return $data;
		}


		/**
		 * compare_vendor_condition function.
		 *
		 * @access public
		 * @param bool $result (default: false), array $cond, array $cart_data
		 * @return bool
		 */
		function compare_vendor_condition( $result, $cond, $cart_data ) {
			// check if type is for Dokan
			if( ! isset( $cond['cond_type'] ) || sanitize_title( $cond['cond_type'] ) != 'dokan_vendor' ) return $result;
			if( ! isset( $cart_data['dokan_vendors'] ) || ! is_array( $cart_data['dokan_vendors'] ) ) return $result;

			// check for empty settings value
			if( ! isset( $cond['cond_tertiary'] ) || empty( $cond['cond_tertiary'] ) ) return $result;

			// compare vendor data
			$array_comp = array_intersect( $cond['cond_tertiary'], $cart_data['dokan_vendors'] );
			if( $cond['cond_secondary'] == 'excludes' && empty( $array_comp ) ) {
				$result = true;
			} elseif( ! empty( $array_comp ) ) {
				$result = true;
			}

			return $result;
		}

	}

	return new BETRS_Dokan();

}

?>