<?php
/*
 * Compatibility Patch for WPC Product Bundles by WPClever
 */

if ( ! defined( 'ABSPATH' ) )
	exit;

// Check if WooCommerce is active
if ( class_exists( 'WooCommerce' ) && class_exists( 'WC_Product_Woosb' ) ) {

	if ( class_exists( 'BETRS_WPC_Bundles' ) ) return;

	class BETRS_WPC_Bundles {

		/**
		 * Cloning is forbidden. Will deactivate prior 'instances' users are running
		 *
		 * @since 4.0
		 */
		public function __clone() {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Cloning this class could cause catastrophic disasters!', 'be-table-ship' ), '4.0' );
		}

		/**
		 * Unserializing instances of this class is forbidden.
		 *
		 * @since 4.0
		 */
		public function __wakeup() {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Unserializing is forbidden!', 'be-table-ship' ), '4.0' );
		}

		/**
		 * __construct function.
		 *
		 * @access public
		 * @return void
		 */
		function __construct() {

			// modify the necessary settings values through hooks and filters
			add_filter( 'betrs_calculated_item-price', array( $this, 'calculate_item_price' ), 10, 3 );

			// disable shipping coupon settings with this plugin
			add_filter( 'woocommerce_shipping_instance_form_fields_betrs_shipping', array( $this, 'disable_coupons_field' ), 20, 1 );
			add_action( 'admin_enqueue_scripts', array( $this, 'custom_styles' ), 999 );
		}


		/**
		 * determine new price based on bundle.
		 *
		 * @access public
		 * @param array $package (default: array())
		 * @return void
		 */
		function calculate_item_price( $price, $item, $method ) {

			// return if no product data
			if( ! is_array( $item ) || ! isset( $item['data'] ) || ! is_object( $item['data'] ) ) return $price;

			// return if not a bundle item
			if( ! method_exists( $item['data'], 'get_type' ) || $item['data']->get_type() != 'woosb' ) return $price;

			// set new price
			$return_cost = floatval( $item['woosb_price'] ) * absint( $item['quantity'] );

			// add coupons to subtotal if enabled
			// coupon inclusion not possible with this extension

			// add tax to subtotal if enabled
			if( $method->includetax === 'yes' ) {
				$return_cost += $this->calculate_item_tax( $return_cost, $item['data'] );
			}

			return $return_cost;
		}


		/**
		 * determine new price based on bundle.
		 *
		 * @access public
		 * @param array $package (default: array())
		 * @return void
		 */
		function calculate_item_tax( $price, $product ) {
			if ( $product->is_taxable() ) {
				// get tax settings
				$tax_rates = WC_Tax::get_rates( $product->get_tax_class() );
				$price_includes_tax = wc_prices_include_tax();

				// calculate tax
				$subtotal_taxes = WC_Tax::calc_tax( $price, $tax_rates, $price_includes_tax );
				$subtotal_tax   = array_sum( array_map( array( $this, 'round_line_tax' ), $subtotal_taxes ) );

				return $subtotal_tax;
			}
		}


		/**
		 * adjust if rounding at subtotal.
		 *
		 * @access public
		 * @param array $package (default: array())
		 * @return void
		 */
		function round_line_tax( $value ) {
			if( 'yes' === get_option( 'woocommerce_tax_round_at_subtotal' ) ) {
				$value = wc_round_tax_total( $value, 0 );
			}

			return $value;
		}


		/**
		 * disable Table Rate coupon price inclusion.
		 *
		 * @access public
		 * @param array $instance_form_fields
		 * @return array $instance_form_fields
		 */
		function disable_coupons_field( $instance_form_fields ) {
			$instance_form_fields['other']['settings']['include_coupons']['disabled'] = true;
			$instance_form_fields['other']['settings']['include_coupons']['label'] = esc_html__( 'Unavailable', 'be-table-ship' );
			$instance_form_fields['other']['settings']['include_coupons']['class'] = 'disabled';
			$instance_form_fields['other']['settings']['include_coupons']['description'] = esc_html__( 'This option cannot be used in combination with the WPC Product Bundles plugin active on this site.', 'be-table-ship' );

			return $instance_form_fields;
		}


		/**
		 * add styling to the disabled coupon option.
		 *
		 * @access public
		 */
		function custom_styles() {
			// add different style to better note the disabled field
		    wp_add_inline_style( 'betrs-dashboard', '.betrs_settings_section label[for=woocommerce_betrs_shipping_include_coupons] { color: #acacac !important; }' );
		}

	}

	new BETRS_WPC_Bundles();

}

?>