<?php
/*
 * Plugin Name: WooCommerce Table Rate Shipping
 * Plugin URI: http://bolderelements.net/plugins/table-rate-shipping-woocommerce/
 * Description: WooCommerce custom plugin designed to calculate shipping costs and add one or more rates based on a table of rules
 * Author: Bolder Elements
 * Author URI: http://www.bolderelements.net/
 * Text Domain: be-table-ship
 * Version: 4.3.9
 * WC requires at least: 4.0.0
 * WC tested up to: 7.0.0

	Copyright: © 2014-2023 Bolder Elements (email : info@bolderelements.net)
	License: Envato Regular License
	License URI: https://codecanyon.net/licenses/terms/regular
*/

if ( ! defined( 'ABSPATH' ) )
	exit;

// Declare compatibility with WooCommerce HPOS
add_action( 'before_woocommerce_init', function() {
	if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
	}
} );

add_action('plugins_loaded', 'woocommerce_table_rate_shipping_init', 0);

function woocommerce_table_rate_shipping_init() {

	//Check if WooCommerce is active
	if ( ! class_exists( 'WooCommerce' ) ) return;

	// Ensure there are not duplicate classes
	if ( class_exists( 'BE_Table_Rate_WC' ) ) return;

	// setup internationalization support
	load_plugin_textdomain('be-table-ship', false, 'woocommerce-table-rate-shipping/languages');

	// include Envato plugin updater file
	include_once( plugin_dir_path( __FILE__ ) . 'inc/envato-market-installer.php' );

	class BE_Table_Rate_WC {

		/*
		 * Version Number
		 */
		protected $version = '4.3.8';

		/*
		 * Table Rates Class
		 */
		public $table_rates;

		/*
		 * Allowed HTML tags for descriptions
		 */
		public $allowedtags;

		/*
		 * Table Rates Class
		 */
		public $method_conditions;

		/*
		 * Allowed HTML tags for descriptions
		 */
		public $user_management;

		/**
		 * Cloning is forbidden. Will deactivate prior 'instances' users are running
		 *
		 * @since 4.0
		 */
		public function __clone() {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Cloning this class could cause catastrophic disasters!', 'be-table-ship' ), '4.0' );
		}

		/**
		 * Unserializing instances of this class is forbidden.
		 *
		 * @since 4.0
		 */
		public function __wakeup() {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Unserializing is forbidden!', 'be-table-ship' ), '4.0' );
		}

		/**
		 * __construct function.
		 *
		 * @access public
		 * @return void
		 */
		function __construct() {
			// Include required files
			if( is_admin() ) {
				// Admin only includes
				add_action( 'admin_enqueue_scripts', array( $this, 'register_plugin_admin' ) );
			}

			add_action( 'woocommerce_init', array( $this, 'includes' ), 0 );
			add_action( 'woocommerce_after_shipping_rate', array( $this, 'display_option_description' ), 10, 2 );
			add_action( 'woocommerce_order_shipping_to_display', array( $this, 'display_option_description_review' ), 10, 2 );
			add_filter( 'woocommerce_shipping_methods', array( $this, 'add_shipping_method' ) );
			add_filter( 'woocommerce_screen_ids', array( $this, 'add_settings_screen' ) );
			add_filter( 'woocommerce_order_item_get_formatted_meta_data', array( $this, 'dashboard_description_fix' ), 10, 2 );
			add_filter( 'manage_edit-shop_order_columns', array( $this, 'add_order_column' ), 10, 1 );
			add_filter( 'default_hidden_columns', array( $this, 'hide_shipping_desc_columns' ), 10, 2 );
			add_filter( 'manage_shop_order_posts_custom_column', array( $this, 'populate_shipping_desc_column' ), 10, 1 );

		}


		/**
		 * setup included files
		 *
		 * @access public
		 * @return void
		 */
		function includes() {
			/**
			 * Description allowed HTML elements.
			 */
			$this->allowedtags = apply_filters( 'betrs_desc_allowed_tags', array(
					'a' => array(
						'href' => true,
						'title' => true,
					),
					'abbr' => array(
						'title' => true,
					),
					'acronym' => array(
						'title' => true,
					),
					'b' => array(),
					'blockquote' => array(
						'cite' => true,
					),
					'br' => array(),
					'cite' => array(),
					'code' => array(),
					'del' => array(
						'datetime' => true,
					),
					'em' => array(),
					'i' => array(),
					'img'     => array(
						'src'   => array(),
						'class' => array(),
						'alt'   => array(),
					),
					'p' => array(
						'align' => true,
						'dir' => true,
						'lang' => true,
						'xml:lang' => true,
					),
					'q' => array(
						'cite' => true,
					),
					's' => array(),
					'span' => array(
						'class' => array(),
						'style' => array(),
						),
					'strike' => array(),
					'strong' => array(),
				)
			);

			// Setup shipping method
			include_once( 'inc/class.shipping-method.php' );
			include_once( 'inc/class.table-rate_options.php' );
			include_once( 'inc/class.calculate-rates.php' );

			// Setup additional settings requirements
			include_once( 'inc/class.settings-shipping-classes.php' );
			include_once( 'inc/class.settings-table-rates.php' );
			$this->table_rates = new BETRS_Table_Rates();

			include_once( 'inc/class.settings-method-conditions.php' );
			$this->method_conditions = new BE_Table_Rate_Method_Conditions();
			$this->user_management = include_once( 'inc/admin/class.user-management.php' );

			// Inclued dashboard only files
			if( is_admin() ) {
			}

			// add compatibility modules
			$this->include_compatibility_modules();
		}


		/**
		 * setup included files
		 *
		 * @access public
		 * @return void
		 */
		function include_compatibility_modules() {
			// Setup compatibility functions
			include_once( 'compatibility/comp.wpml.php' );
			include_once( 'compatibility/comp.wcml.php' );
			include_once( 'compatibility/comp.currency_switcher.php' );
			include_once( 'compatibility/comp.wmc.php' );
			include_once( 'compatibility/comp.polylang.php' );
			include_once( 'compatibility/comp.dokan.php' );
			include_once( 'compatibility/comp.measurement_price_calc.php' );
			include_once( 'compatibility/comp.wpc_bundles.php' );
		}



		/**
		 * Return version number for others to view
		 *
		 * @access public
		 * @return string
		 */
		public function get_version() {

			return $this->version;
		}

		/**
		 * add_cart_rate_method function.
		 *
		 * @package		WooCommerce/Classes/Shipping
		 * @access public
		 * @param array $methods
		 * @return array
		 */
		function add_shipping_method( $methods ) {
			$methods['betrs_shipping'] = 'BE_Table_Rate_Method';

			return $methods;
		}


	    /**
	     * display description if applicable.
	     *
	     * @access public
	     * @param mixed $method
	     * @return void
	     */
	    function display_option_description( $method, $index ) {
	    	global $betrs_shipping;

	    	$meta_data = $method->get_meta_data();
	    	if( isset( $meta_data['description'] ) && is_string( $meta_data['description'] ) ) {
	    		$desc = stripslashes( wp_kses( $meta_data['description'], $betrs_shipping->allowedtags ) );
	    	} elseif( isset( $meta_data['description'] ) && is_object( $meta_data['description'] ) ) {
	    		$desc = stripslashes( wp_kses( $meta_data['description']->scalar, $betrs_shipping->allowedtags ) );
	    	}

	    	if( isset( $desc ) )
	    		echo '<div class="betrs_option_desc">' . do_shortcode( $desc ) . '</div>';
	    }


	    /**
	     * display description if applicable.
	     *
	     * @access public
	     * @param mixed $method
	     * @return void
	     */
	    function display_option_description_review( $shipping, $order ) {
		    global $betrs_shipping;
		    
		    foreach( $order->get_shipping_methods() as $method ) {
		    	if( ! is_object( $method ) || ! method_exists( $method, 'get_meta_data' ) )
		    		continue;

		        $meta_data = $method->get_meta_data();
		        foreach( $meta_data as $meta ) {
		        	if( method_exists( $meta, 'get_data' ) ) {
			            $data = $meta->get_data();
			            if( $data['key'] == 'description' && ! empty( $data['value'] ) ) {
							$desc = ( is_object( $data['value'] ) ) ? $data['value']->scalar : $data['value'];
	    					$desc = stripslashes( wp_kses( $desc, $betrs_shipping->allowedtags ) );
			            	$shipping .= '<div class="betrs_option_desc betrs_order_review">' . do_shortcode( $desc ) . '</div>';
			            }
			    	}
		        }
		    }
		    return $shipping;
		}


	    /**
	     * fix dashboard meta data display for orders prior to 4.3.
	     *
	     * @access public
	     * @param mixed $method
	     * @return void
	     */
	    function dashboard_description_fix( $formatted_meta, $item ) {
	    	global $betrs_shipping;
	    	
	    	// check shipping meta for description type
			if( $item->is_type('shipping') ) {
				$shipping_meta = $item->get_meta_data();

		    	if( is_array( $shipping_meta ) ) {
		    		foreach( $shipping_meta as $key => $value ) {
		    			$meta_data = $value->get_data();

		    			if( $meta_data['key'] == 'description' && is_object( $meta_data['value'] ) ) {
		    				$new_meta = new stdClass();
		    				$new_meta->key				= "description";
		    				$new_meta->value			= wp_kses( $meta_data['value']->scalar, $betrs_shipping->allowedtags );
		    				$new_meta->display_key		= esc_html__( 'Description', 'be-table-ship' );
		    				$new_meta->display_value	= wpautop( make_clickable( apply_filters( 'woocommerce_order_item_display_meta_value', $meta_data['value']->scalar, $meta_data, $item ) ) );

		    				$formatted_meta[ $meta_data['id'] ] = $new_meta;
		    			}
		    		}
		    	}				
			}

			return $formatted_meta;
	    }


	    /**
		 * Adds 'Shipping Description' column 'Orders' page.
		 *
		 * @param string[] $columns
		 * @return string[] $new_columns
		 */
		function add_order_column( $columns ) {

		    $new_columns = array();

		    foreach ( $columns as $column_name => $column_info ) {

		        $new_columns[ $column_name ] = $column_info;

		        if ( 'order_total' === $column_name ) {
		            $new_columns['shipping_desc'] = esc_html__( 'Shipping Description', 'be-table-ship' );
		        }
		    }

		    return $new_columns;
		}


	    /**
		 * Hides new 'Shipping Description' column by default.
		 *
		 * @param string[] $hidden
		 * @param object $screen
		 * @return string[] $new_columns
		 */
		function hide_shipping_desc_columns( $hidden, $screen ) {

		    if( isset( $screen->id ) && 'edit-shop_order' === $screen->id ){      
		        $hidden[] = 'shipping_desc';     
		    }   
		    return $hidden;
		}


		/**
		* Adds 'Profit' column content to 'Orders' page immediately after 'Total' column.
		*
		* @param string[] $column name of column being displayed
		*/
		function populate_shipping_desc_column( $column ) {
			global $post, $betrs_shipping;

			if ( 'shipping_desc' === $column ) {

				$order    = wc_get_order( $post->ID );
				$shipping = $order->get_items('shipping');

				// check if a description is available
				if ( is_array( $shipping ) && count( $shipping ) ) {

					foreach( $shipping as $shipping_op ) {

						if( $shipping_op->get_method_id() == 'betrs_shipping' ) {

							$meta_data = $shipping_op->get_meta_data();

							if( is_array( $meta_data ) && count( $meta_data ) ) {

								foreach( $meta_data as $meta ) {

									$current_data = $meta->get_data();

									if( sanitize_title( $current_data['key'] ) == 'description' ) {

										if( is_object( $current_data['value'] ) ) {
											echo stripslashes( wp_kses( $current_data['value']->scalar, $betrs_shipping->allowedtags ) );
										} else {
											echo stripslashes( wp_kses( $current_data['value'], $betrs_shipping->allowedtags ) );
										}
									}
								}
							}
						}
					}
				}
			}
		}


        /**
         * add settings page to list of WC settings
		 *
		 * @access public
		 * @return void
         */
		public function add_settings_screen( $current_screens ) {

			$current_screens[] = 'toplevel_page_betrs-manage-shipping';
			return $current_screens;
		}



		/**
		 * Modify Scripts in Dashboard
		 *
		 * @access public
		 * @return void
		 */
		public function register_plugin_admin( $hook_suffix ) {
			global $current_screen;
			
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			$ver = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? null : $this->version;

			// only load on WooCommerce settings page
			$wc_screen_id = sanitize_title( strtolower( esc_html__( 'WooCommerce', 'woocommerce' ) ) );
			$accepted_pages = array( $wc_screen_id . '_page_wc-settings', 'toplevel_page_betrs-manage-shipping' );
			if( $current_screen && ( ! in_array( $current_screen->id, $accepted_pages ) ) ) return;

			// JS includes
			wp_enqueue_script( 'jquery-ui-datepicker' );
			wp_enqueue_script( 'betrs-settings', plugins_url( 'assets/js/settings' . $suffix . '.js', __FILE__ ), array( 'jquery' ), $ver, true );
			wp_enqueue_script( 'betrs-settings-table-rates', plugins_url( 'assets/js/settings.table-rates' . $suffix . '.js', __FILE__ ), array( 'jquery', 'jquery-ui-datepicker' ), $ver, true );

			// CSS includes
			wp_enqueue_style( 'betrs-dashboard', plugins_url( 'assets/css/dashboard.css', __FILE__ ), false, $ver );

			// add localized strings to JS file
			$betrs_data = array(
				'ajax_url'					=> addcslashes( admin_url( 'admin-ajax.php', 'relative' ), '/' ),
				'ajax_loader_url'			=> plugins_url( 'assets/img/loader.gif', __FILE__ ),
				'text_ok'					=> esc_html__( 'OK', 'be-table-ship' ),
				'text_edit'					=> esc_html__( 'Edit', 'be-table-ship' ),
				'text_error'				=> esc_html__( 'Error', 'be-table-ship' ),
				'text_upload'				=> esc_html__( 'Upload', 'be-table-ship' ),
				'text_cancel'				=> esc_html__( 'Cancel', 'be-table-ship' ),
				'text_delete_confirmation'	=> esc_html__( 'Are you sure you want to do this? Delete actions cannot be undone.', 'be-table-ship' ),
				'text_importing_table'		=> esc_html__( 'Import Table of Rates', 'be-table-ship' ),
				'text_importing_csv'		=> esc_html__( 'Select a CSV file', 'be-table-ship' ),
				'text_importing_del'		=> esc_html__( 'Delete existing rows before importing', 'be-table-ship' ),
				'text_exporting'			=> esc_html__( 'Exporting', 'be-table-ship' ),
				'text_no_selection'			=> esc_html__( 'Please select which rows you wish to export', 'be-table-ship' ),
				'text_error_server'			=> esc_html__( 'Please review the documentation and enable the required server settings', 'be-table-ship' ),
				);
			$encoded_data = json_encode( $betrs_data );
			wp_add_inline_script( 'betrs-settings', 'var betrs_data = ' . $encoded_data, 'before' );
		}


	} /* End Class BE_Table_Rate_WC */

	$GLOBALS['betrs_shipping'] = new BE_Table_Rate_WC();

} // End woocommerce_table_rate_shipping_init.
 

/**
 * Add links to dashboard Plugins page
 *
 * @access public
 * @return void
 */
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'be_table_rate_wc_action_links' );
function be_table_rate_wc_action_links( $links ) {
	return array_merge(
		array(
			'settings' => '<a href="' . get_admin_url() . '/admin.php?page=wc-settings&tab=shipping">' . esc_html__( 'Settings', 'be-table-ship' ) . '</a>',
			'support' => '<a href="http://bolderelements.net/support/" target="_blank">' . esc_html__( 'Support', 'be-table-ship' ) . '</a>'
		),
		$links
	);
 
}


/**
 * API check.
 *
 * @since 4.2.0
 *
 * @param bool   $api Always false.
 * @param string $action The API action being performed.
 * @param object $args Plugin arguments.
 * @return mixed $api The plugin info or false.
 */
function betrs_override_plugins_api_result( $res, $action, $args ) {
	
	if ( isset( $args->slug ) && 'woocommerce-table-rate-shipping' === $args->slug ) {
		$api_check = betrs_override_api_check();
		if ( is_object( $api_check ) ) {
			$res = $api_check;
			$res->external = true;
		}
	}

	return $res;
}
add_filter( 'plugins_api_result', 'betrs_override_plugins_api_result', 10, 3 );


/**
 * Check Github for an update.
 *
 * @since 4.2.0
 *
 * @return false|object
 */
function betrs_override_api_check() {
	$raw_response = wp_remote_get( 'https://bolderelements.github.io/docs/woocommerce-table-rate-shipping/update-check.json' );

	if ( is_wp_error( $raw_response ) ) {
		return false;
	}
	if ( ! empty( $raw_response['body'] ) ) {
		$raw_body = json_decode( trim( $raw_response['body'] ), true );
		if ( $raw_body ) {
			return (object) $raw_body;
		}
	}
	return false;
}

?>